<?php

namespace Drupal\library_renderer;

use Drupal\Core\Asset\LibraryDiscoveryParser;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Theme\Registry;
use Drupal\Core\Theme\ThemeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Provides library renderer for attaching libraries via YML.
 */
class LibraryRenderer {

  /**
   * An array of template YAML definitions.
   *
   * @var array
   */
  private array $parsedLibraries;

  /**
   * An array of hooked libraries.
   *
   * @var array
   */
  private array $hookedLibraries;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The theme manager.
   *
   * @var \Drupal\Core\Theme\ThemeManagerInterface
   */
  protected ThemeManagerInterface $themeManager;

  /**
   * The library discovery parser.
   *
   * @var \Drupal\Core\Asset\LibraryDiscoveryParser
   */
  protected LibraryDiscoveryParser $discoveryParser;

  /**
   * The theme registry used to determine which template to use.
   *
   * @var \Drupal\Core\Theme\Registry
   */
  protected Registry $themeRegistry;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new ComponentLibraryParser object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Theme\ThemeManagerInterface $theme_manager
   *   The theme manager.
   * @param \Drupal\Core\Asset\LibraryDiscoveryParser $discovery_parser
   *   The library discovery parser.
   * @param \Drupal\Core\Theme\Registry $theme_registry
   *   The theme registry.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   */
  public function __construct(ModuleHandlerInterface $module_handler, ThemeManagerInterface $theme_manager, LibraryDiscoveryParser $discovery_parser, Registry $theme_registry, LoggerChannelFactoryInterface $logger_factory) {
    $this->moduleHandler = $module_handler;
    $this->themeManager = $theme_manager;
    $this->discoveryParser = $discovery_parser;
    $this->themeRegistry = $theme_registry;
    $this->logger = $logger_factory->get('library_renderer');
    $this->getParsedLibraries();
    $this->getHookedLibraries();
  }

  /**
   * Attach libraries to $variables.
   */
  public function attachLibraries(array &$variables, string $hook, array $info): void {
    try {
      // Attach theme hook libraries defined with theme_hooks YML definition.
      $this->attachThemeHookLibraries($hook, $variables);
      // Get valid suggestions using service decorator.
      $suggestions = $this->themeManager->getThemeHookSuggestions($hook, $info['base hook'] ?? '', $variables);
      $valid_suggestions = $this->getValidSuggestions($hook, $info['base hook'] ?? '', $suggestions);
      if (!empty($valid_suggestions)) {
        // Apply libraries defined with theme_suggestions YML definition.
        $this->attachThemeSuggestionLibraries($variables, $valid_suggestions);
        // Apply libraries defined with templates YML definition.
        $this->attachTemplateLibraries($hook, $variables, $valid_suggestions);
      }
    }
    catch (\Throwable $exception) {
      $this->logger->info($exception->getMessage());
    }
  }

  /**
   * Get the parsed libraries.
   */
  public function getParsedLibraries(): array {
    if (empty($this->parsedLibraries)) {
      $this->parsedLibraries = $this->generateParsedLibraries();
    }
    return $this->parsedLibraries;
  }

  /**
   * Get the hooked libraries.
   */
  public function getHookedLibraries(): array {
    if (empty($this->hookedLibraries)) {
      $this->hookedLibraries = $this->generateHookedLibraries();
    }
    return $this->hookedLibraries;
  }

  /**
   * Generates parsed libraries.
   *
   * @return array
   *   The parsed libraries.
   */
  public function generateParsedLibraries(): array {
    $active_theme = $this->themeManager->getActiveTheme();
    $parsedLibraries[$active_theme->getName()] = $this->discoveryParser->buildByExtension($active_theme->getName());
    // Parse module libraries.
    $modules = $this->moduleHandler->getModuleList();
    if (!empty($modules)) {
      foreach ($modules as $module_key => $module) {
        $parsedLibraries[$module_key] = $this->discoveryParser->buildByExtension($module_key);
      }
    }
    return $parsedLibraries;
  }

  /**
   * Generate hooked libraries.
   */
  public function generateHookedLibraries(): array {
    if (empty($this->parsedLibraries)) {
      $this->getParsedLibraries();
    }
    $hookedLibraries = [];
    $rendererTypes = [
      'html_tags',
      'templates',
      'theme_hooks',
      'theme_suggestions',
    ];
    foreach ($this->parsedLibraries as $extension => $libraries) {
      if (!empty($libraries)) {
        foreach ($libraries as $library_key => $library) {
          if (!empty($library['library_renderer'])) {
            foreach ($library['library_renderer'] as $library_renderer_type => $library_renderer) {
              if ($library_renderer_type === 'html_tags') {
                foreach ($library_renderer as $html_tag => $attributes) {
                  $hookedLibraries[$library_renderer_type][$html_tag][] = [
                    'library' => $extension . '/' . $library_key,
                  ] + $attributes;
                }
                continue;
              }
              if (in_array($library_renderer_type, $rendererTypes, TRUE)) {
                foreach ($library_renderer as $hook) {
                  $hookedLibraries[$library_renderer_type][$hook][] = $extension . '/' . $library_key;
                }
              }
            }
          }
        }
      }
    }
    return $hookedLibraries;
  }

  /**
   * Attach hooked libraries.
   */
  public function attachThemeHookLibraries($hook, &$variables): void {
    if (empty($this->hookedLibraries['theme_hooks'])) {
      return;
    }
    // Iterate over the theme hooks with those defined in libraries.yml.
    if (array_key_exists($hook, $this->hookedLibraries['theme_hooks'])) {
      foreach ($this->hookedLibraries['theme_hooks'] as $key => $libraries) {
        if (!empty($libraries)) {
          foreach ($libraries as $library) {
            if ($key === $hook) {
              $variables['#attached']['library'][] = $library;
            }
          }
        }
      }
    }
  }

  /**
   * Attach hooked libraries.
   */
  public function attachThemeSuggestionLibraries(&$variables, $valid_suggestions): void {
    if (empty($this->hookedLibraries['theme_suggestions'])) {
      return;
    }
    // Iterate over the suggested hooks with those defined in libraries.yml.
    foreach ($this->hookedLibraries['theme_suggestions'] as $key => $libraries) {
      if (in_array($key, $valid_suggestions)) {
        foreach ($libraries as $library) {
          $variables['#attached']['library'][] = $library;
        }
      }
    }
  }

  /**
   * Attach hooked libraries.
   */
  public function attachTemplateLibraries($hook, &$variables, $valid_suggestions): void {
    if (empty($this->hookedLibraries['templates'])) {
      return;
    }
    $active_template = $this->getActiveTemplate($hook, $valid_suggestions);
    // Iterate over the suggested hooks with those defined in libraries.yml.
    if (array_key_exists($active_template, $this->hookedLibraries['templates'])) {
      foreach ($this->hookedLibraries['templates'] as $template => $libraries) {
        if (!empty($libraries)) {
          foreach ($libraries as $library) {
            if ($template === $active_template) {
              $variables['#attached']['library'][] = $library;
            }
          }
        }
      }
    }
  }

  /**
   * Valid suggestions are $base_hook, $base_hook__*, and contain no hyphens.
   *
   * See: twig_render_template().
   */
  public function getValidSuggestions(string $hook, string $info_base_hook, array $suggestions): array {
    $base_hook = !empty($info_base_hook) ? $info_base_hook : $hook;
    if (!str_contains($base_hook, '__') && !in_array($base_hook, $suggestions)) {
      $suggestions[] = $base_hook;
    }
    foreach ($suggestions as $key => &$suggestion) {
      if (($suggestion !== $base_hook && !str_starts_with($suggestion, $base_hook . '__')) || str_contains($suggestion, '-')) {
        unset($suggestions[$key]);
      }
    }
    return $suggestions;
  }

  /**
   * Gets active template.
   *
   * See: twig_render_template().
   */
  public function getActiveTemplate($hook, $valid_suggestions): string {
    $info = $this->themeRegistry->getRuntime()->get($hook);
    $engine = $this->themeManager->getActiveTheme()->getEngine();
    $extension = '.html.twig';
    // Check if each suggestion exists in the theme registry, and if so,
    // use it instead of the base hook. For example, a function may use
    // '#theme' => 'node', but a module can add 'node__article' as a suggestion
    // via hook_theme_suggestions_HOOK_alter(), enabling a theme to have
    // an alternate template file for article nodes.
    $theme_registry = $this->themeRegistry->getRuntime();
    foreach ($valid_suggestions as $suggestion) {
      if ($theme_registry->has($suggestion)) {
        $info = $theme_registry->get($suggestion);
        break;
      }
    }
    if (isset($engine)) {
      if ($info['type'] != 'module') {
        $extension_function = $engine . '_extension';
        if (function_exists($extension_function)) {
          $extension = $extension_function();
        }
      }
    }
    // Render the output using the template file.
    $template_file = $info['template'] . $extension;
    if (isset($info['path'])) {
      $template_file = $info['path'] . '/' . $template_file;
    }
    $current_template = basename($template_file);
    $active_template = '';
    foreach ($valid_suggestions as $key => &$suggestion) {
      $template = strtr($suggestion, '_', '-') . $extension;
      if ($current_template == $template) {
        $active_template = $template;
      }
    }
    return $active_template;
  }

}
