/**
 * @file
 * Javascript behaviors for LightGallery integration.
 */

/* global lightGallery, lgThumbnail */

(function ($, Drupal, drupalSettings, once) {
  /**
   * Attach LightGallery behavior to matched elements.
   */
  Drupal.behaviors.lightgalleryFormatter = {
    attach(context, settings) {
      const galleries = drupalSettings.lightgallery_formatter?.galleries;
      if (!galleries) {
        return;
      }

      // Process each gallery configuration.
      Object.keys(galleries).forEach((galleryId) => {
        const galleryConfig = galleries[galleryId];
        const galleryElements = once('lightgallery', `#${galleryId}`, context);

        galleryElements.forEach((element) => {
          // Check if lightGallery is available.
          if (typeof lightGallery === 'undefined') {
            console.warn('LightGallery library is not loaded.');
            return;
          }

          // Delay initialization to allow DOM to render.
          setTimeout(() => {
            try {
              // Prepare plugins array.
              const plugins = [];

              // Add thumbnail plugin if enabled and available.
              if (
                galleryConfig.thumbnail &&
                typeof lgThumbnail !== 'undefined'
              ) {
                plugins.push(lgThumbnail);
              }

              // Build settings from configuration and data attributes.
              const lgSettings = {
                ...{
                  selector: 'a',
                  licenseKey: '0000-0000-000-0000',
                  plugins,
                  download: galleryConfig.download ?? false,
                  loop: galleryConfig.loop ?? true,
                  preload: parseInt(galleryConfig.preload, 10) || 2,
                  speed: galleryConfig.speed ?? 400,
                  mode: galleryConfig.mode ?? 'lg-slide',
                  closable: galleryConfig.closable ?? true,
                  closeOnTap: galleryConfig.closeOnTap ?? true,
                  escKey: galleryConfig.escKey ?? true,
                  keyPress: galleryConfig.keyPress ?? true,
                  showCloseIcon: galleryConfig.showCloseIcon ?? true,
                  showMaximizeIcon: galleryConfig.showMaximizeIcon ?? false,
                  hideBarsDelay: galleryConfig.hideBarsDelay ?? 0,
                  counter: galleryConfig.counter ?? true,
                  enableDrag: galleryConfig.enableDrag ?? true,
                  enableSwipe: galleryConfig.enableSwipe ?? true,
                  mousewheel: galleryConfig.mousewheel ?? false,
                },
                ...galleryConfig,
              };

              // Initialize LightGallery.
              lightGallery(element, lgSettings);
            } catch (e) {
              console.error('Error initializing LightGallery:', e);
            }
          }, 100);
        });
      });
    },
  };
})(jQuery, Drupal, drupalSettings, once);
