<?php

namespace Drupal\lightgallery_formatter\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ExtensionPathResolver;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryThumbnailSettings;

/**
 * Form handler for the LightGallery profile add and edit forms.
 */
class LightgalleryProfileForm extends EntityForm {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The extension path resolver.
   *
   * @var \Drupal\Core\Extension\ExtensionPathResolver
   */
  protected $extensionPathResolver;

  /**
   * The module extension list.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * The app root path.
   *
   * @var string
   */
  protected $appRoot;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The preview builder service (if available).
   *
   * @var \Drupal\lightgallery_formatter_preview\PreviewBuilder|null
   */
  protected $previewBuilder;

  /**
   * Constructs a new LightgalleryProfileForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ExtensionPathResolver $extension_path_resolver
   *   The extension path resolver.
   * @param \Drupal\Core\Extension\ModuleExtensionList $module_extension_list
   *   The module extension list.
   * @param string $app_root
   *   The app root path.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    ExtensionPathResolver $extension_path_resolver,
    ModuleExtensionList $module_extension_list,
    string $app_root,
    ModuleHandlerInterface $module_handler,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->extensionPathResolver = $extension_path_resolver;
    $this->moduleExtensionList = $module_extension_list;
    $this->appRoot = $app_root;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    // @phpstan-ignore-next-line new.static is standard Drupal DI pattern.
    $instance = new static(
      $container->get('entity_type.manager'),
      $container->get('extension.path.resolver'),
      $container->get('extension.list.module'),
      $container->getParameter('app.root'),
      $container->get('module_handler')
    );

    // Inject preview builder if the preview module is enabled.
    if ($container->get('module_handler')->moduleExists('lightgallery_formatter_preview')) {
      $instance->previewBuilder = $container->get('lightgallery_formatter_preview.preview_builder');
    }

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function init(FormStateInterface $form_state) {
    // Ensure $this->entity is set before calling parent::init().
    if (!$this->entity) {
      $storage = $this->entityTypeManager->getStorage('lightgallery_profile');
      $this->entity = $storage->create([
        'status' => TRUE,
      ]);
    }

    // Now safe to call parent.
    parent::init($form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);
    $profile = $this->entity;

    // Preview section at the top of the form.
    $form['preview'] = [
      '#type' => 'details',
      '#title' => $this->t('Gallery Preview'),
      '#open' => TRUE,
      '#weight' => -100,
      '#attributes' => [
        'id' => 'lightgallery-preview-wrapper',
        'class' => ['lightgallery-preview-section'],
      ],
    ];

    $form['preview']['description'] = [
      '#type' => 'markup',
      '#markup' => '<p class="lightgallery-preview-description">' . $this->t('Click "Generate Preview" to see how the gallery will look with current settings.') . '</p>',
    ];

    $form['preview']['generate_button'] = [
      '#type' => 'button',
      '#value' => $this->t('Generate Preview'),
      '#ajax' => [
        'callback' => '::generatePreview',
        'wrapper' => 'lightgallery-preview-content',
        'effect' => 'fade',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Generating preview...'),
        ],
      ],
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
    ];

    $form['preview']['content'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'lightgallery-preview-content',
        'class' => ['lightgallery-preview-content'],
      ],
    ];

    // Check if we need to show the preview (after AJAX callback).
    if ($form_state->get('show_preview')) {
      $form['preview']['content']['gallery'] = $this->buildPreviewGallery($form_state);
    }
    else {
      $form['preview']['content']['placeholder'] = [
        '#type' => 'markup',
        '#markup' => '<div class="lightgallery-preview-placeholder">' . $this->t('Preview will appear here after clicking "Generate Preview".') . '</div>',
      ];
    }

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#description' => $this->t('The human-readable name for this profile.'),
      '#default_value' => $profile->label(),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#title' => $this->t('Machine name'),
      '#description' => $this->t('A unique name for this profile. Can only contain lowercase letters, numbers, and underscores.'),
      '#default_value' => $profile->id(),
      '#disabled' => !$profile->isNew(),
      '#machine_name' => [
        'exists' => '\Drupal\lightgallery_formatter\Entity\LightgalleryProfile::load',
        'source' => ['label'],
      ],
      '#required' => TRUE,
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $profile->getStatus(),
    ];

    $form['use_cdn'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use CDN'),
      '#default_value' => $profile->useCdn(),
      '#description' => $this->t('Load LightGallery files from a CDN instead of locally.'),
      '#ajax' => [
        'callback' => '::updateCdnUrls',
        'wrapper' => 'cdn-urls-wrapper',
        'effect' => 'fade',
      ],
    ];

    $form['cdn_provider'] = [
      '#type' => 'select',
      '#title' => $this->t('CDN Provider'),
      '#options' => [
        'jsdelivr' => $this->t('jsDelivr'),
        'cdnjs' => $this->t('CDNJS'),
        'unpkg' => $this->t('UNPKG'),
      ],
      '#default_value' => $profile->getCdnProvider(),
      '#states' => [
        'visible' => [
          ':input[name="use_cdn"]' => ['checked' => TRUE],
        ],
      ],
      '#description' => $this->t('Select which CDN provider to use for loading LightGallery files.'),
      '#ajax' => [
        'callback' => '::updateCdnUrls',
        'wrapper' => 'cdn-urls-wrapper',
        'effect' => 'fade',
      ],
    ];

    // Display CDN URLs when CDN is enabled.
    $form['cdn_urls'] = [
      '#type' => 'details',
      '#title' => $this->t('CDN URLs'),
      '#open' => TRUE,
      '#attributes' => [
        'id' => 'cdn-urls-wrapper',
      ],
      '#states' => [
        'visible' => [
          ':input[name="use_cdn"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $cdn_provider = $form_state->getValue('cdn_provider') ?? $profile->getCdnProvider();
    $use_cdn = $form_state->getValue('use_cdn') ?? $profile->useCdn();

    if ($use_cdn) {
      $cdn_info = $this->getCdnInfo($cdn_provider);

      $form['cdn_urls']['js_url'] = [
        '#type' => 'markup',
        '#markup' => '<div class="cdn-url-info"><strong>JavaScript:</strong><br><code>' . $cdn_info['js'] . '</code></div>',
      ];

      $form['cdn_urls']['css_url'] = [
        '#type' => 'markup',
        '#markup' => '<div class="cdn-url-info"><strong>CSS:</strong><br><code>' . $cdn_info['css'] . '</code></div>',
      ];
    }
    else {
      $form['cdn_urls']['message'] = [
        '#type' => 'markup',
        '#markup' => '<div class="cdn-url-info">' . $this->t('Enable CDN to see the URLs.') . '</div>',
      ];
    }

    // Get validated general settings (includes defaults).
    $general_settings = $profile->getValidatedPluginSettings('general');

    $form['general_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('General Settings'),
      '#open' => TRUE,
    ];

    $form['general_settings']['loop'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable looping'),
      '#default_value' => $general_settings['loop'],
      '#description' => $this->t('Allow the gallery to loop from last to first image.'),
    ];

    $form['general_settings']['preload'] = [
      '#type' => 'number',
      '#title' => $this->t('Preload count'),
      '#default_value' => $general_settings['preload'],
      '#min' => 1,
      '#max' => 5,
      '#description' => $this->t('Number of images to load in advance.'),
    ];

    $form['general_settings']['speed'] = [
      '#type' => 'number',
      '#title' => $this->t('Transition speed (ms)'),
      '#default_value' => $general_settings['speed'],
      '#min' => 0,
      '#description' => $this->t('Duration of transition between images in milliseconds.'),
    ];

    $form['general_settings']['mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Transition mode'),
      '#default_value' => $general_settings['mode'],
      '#options' => [
        'lg-slide' => $this->t('Slide'),
        'lg-fade' => $this->t('Fade'),
        'lg-zoom-in' => $this->t('Zoom in'),
        'lg-zoom-out' => $this->t('Zoom out'),
        'lg-zoom-out-big' => $this->t('Zoom out big'),
        'lg-zoom-out-in' => $this->t('Zoom out in'),
        'lg-zoom-in-out' => $this->t('Zoom in out'),
        'lg-soft-zoom' => $this->t('Soft zoom'),
        'lg-scale-up' => $this->t('Scale up'),
        'lg-slide-circular' => $this->t('Slide circular'),
        'lg-slide-circular-vertical' => $this->t('Slide circular vertical'),
        'lg-slide-vertical' => $this->t('Slide vertical'),
        'lg-slide-vertical-growth' => $this->t('Slide vertical growth'),
        'lg-slide-skew-only' => $this->t('Slide skew only'),
        'lg-slide-skew-only-rev' => $this->t('Slide skew only reverse'),
        'lg-slide-skew-only-y' => $this->t('Slide skew only Y'),
        'lg-slide-skew-only-y-rev' => $this->t('Slide skew only Y reverse'),
        'lg-slide-skew' => $this->t('Slide skew'),
        'lg-slide-skew-rev' => $this->t('Slide skew reverse'),
        'lg-slide-skew-cross' => $this->t('Slide skew cross'),
        'lg-slide-skew-cross-rev' => $this->t('Slide skew cross reverse'),
        'lg-slide-skew-ver' => $this->t('Slide skew vertical'),
        'lg-slide-skew-ver-rev' => $this->t('Slide skew vertical reverse'),
        'lg-slide-skew-ver-cross' => $this->t('Slide skew vertical cross'),
        'lg-slide-skew-ver-cross-rev' => $this->t('Slide skew vertical cross reverse'),
        'lg-lollipop' => $this->t('Lollipop'),
        'lg-lollipop-rev' => $this->t('Lollipop reverse'),
        'lg-rotate' => $this->t('Rotate'),
        'lg-rotate-rev' => $this->t('Rotate reverse'),
        'lg-tube' => $this->t('Tube'),
      ],
      '#description' => $this->t('Type of transition between images.'),
    ];

    $form['general_settings']['closable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Closable'),
      '#default_value' => $general_settings['closable'],
      '#description' => $this->t("If disabled, users won't be able to close the gallery. Useful for inline galleries."),
    ];

    $form['general_settings']['closeOnTap'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Close on tap'),
      '#default_value' => $general_settings['closeOnTap'],
      '#description' => $this->t('Allows clicks on black area to close gallery.'),
    ];

    $form['general_settings']['escKey'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Close with Esc key'),
      '#default_value' => $general_settings['escKey'],
      '#description' => $this->t('Whether the gallery can be closed by pressing the "Esc" key.'),
    ];

    $form['general_settings']['keyPress'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Keyboard navigation'),
      '#default_value' => $general_settings['keyPress'],
      '#description' => $this->t('Enable keyboard navigation (arrow keys, etc.).'),
    ];

    $form['general_settings']['showCloseIcon'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show close icon'),
      '#default_value' => $general_settings['showCloseIcon'],
      '#description' => $this->t('Display the close button in the gallery.'),
    ];

    $form['general_settings']['showMaximizeIcon'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show maximize icon'),
      '#default_value' => $general_settings['showMaximizeIcon'],
      '#description' => $this->t('Display the maximize button in the gallery.'),
    ];

    $form['general_settings']['hideBarsDelay'] = [
      '#type' => 'number',
      '#title' => $this->t('Hide controls delay (ms)'),
      '#default_value' => $general_settings['hideBarsDelay'],
      '#min' => 0,
      '#description' => $this->t('Delay in milliseconds before hiding gallery controls after interaction. Set to 0 to keep controls visible.'),
    ];

    $form['general_settings']['counter'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show counter'),
      '#default_value' => $general_settings['counter'],
      '#description' => $this->t('Display the total number of images and index of the currently displayed image.'),
    ];

    $form['general_settings']['download'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable download'),
      '#default_value' => $general_settings['download'],
      '#description' => $this->t('Display the download button for images.'),
    ];

    $form['general_settings']['enableDrag'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable desktop drag'),
      '#default_value' => $general_settings['enableDrag'],
      '#description' => $this->t('Enable mouse drag support for desktop devices.'),
    ];

    $form['general_settings']['enableSwipe'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable touch swipe'),
      '#default_value' => $general_settings['enableSwipe'],
      '#description' => $this->t('Enable swipe support for touch devices.'),
    ];

    $form['general_settings']['mousewheel'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Mouse wheel navigation'),
      '#default_value' => $general_settings['mousewheel'],
      '#description' => $this->t('Enable navigation to next/prev slides using mouse wheel.'),
    ];

    // Enable thumbnail settings checkbox.
    $current_thumbnail_settings = $profile->getPluginSettings('thumbnail');
    $thumbnail_enabled = !empty($current_thumbnail_settings);

    $form['enable_thumbnail'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable thumbnail plugin'),
      '#default_value' => $thumbnail_enabled,
      '#description' => $this->t('Check this to configure thumbnail navigation settings.'),
    ];

    // Thumbnail plugin settings (conditionally visible)
    $thumbnail_plugin = new LightgalleryThumbnailSettings();
    $default_settings = $thumbnail_plugin->getDefaultSettings();
    $current_settings = $profile->getPluginSettings('thumbnail');

    $form['thumbnail_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Thumbnail Settings'),
      '#open' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="enable_thumbnail"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['thumbnail_settings']['animateThumb'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Animate thumbnails'),
      '#default_value' => $current_settings['animateThumb'] ?? $default_settings['animateThumb'],
      '#description' => $this->t('Animate thumbnail navigation.'),
    ];

    $form['thumbnail_settings']['showThumbByDefault'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show thumbnails by default'),
      '#default_value' => $current_settings['showThumbByDefault'] ?? $default_settings['showThumbByDefault'],
      '#description' => $this->t('Show thumbnails when the gallery loads.'),
    ];

    // Get available image styles.
    $image_styles = ['' => $this->t('- None -')];
    foreach ($this->entityTypeManager->getStorage('image_style')->loadMultiple() as $image_style) {
      $image_styles[$image_style->id()] = $image_style->label();
    }

    $form['thumbnail_settings']['image_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Thumbnail image style'),
      '#description' => $this->t('Select the image style to use for thumbnails. If none is selected, the original image will be used.'),
      '#options' => $image_styles,
      '#default_value' => $current_settings['image_style'] ?? $default_settings['image_style'],
    ];

    $form['thumbnail_settings']['thumbMargin'] = [
      '#type' => 'number',
      '#title' => $this->t('Thumbnail margin'),
      '#default_value' => $current_settings['thumbMargin'] ?? $default_settings['thumbMargin'],
      '#min' => 0,
      '#description' => $this->t('Margin between thumbnails in pixels.'),
    ];

    $form['thumbnail_settings']['appendTo'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Append to'),
      '#default_value' => $current_settings['appendTo'] ?? $default_settings['appendTo'],
      '#description' => $this->t('CSS selector of the element where thumbnails will be appended.'),
    ];

    return $form;
  }

  /**
   * Get CDN URLs for a given provider by parsing the libraries.yml file.
   *
   * @param string $provider
   *   The CDN provider (jsdelivr, cdnjs, unpkg).
   *
   * @return array
   *   Associative array with 'js' and 'css' keys containing the URLs.
   */
  protected function getCdnInfo($provider) {
    // Map form provider values to library names.
    $library_map = [
      'jsdelivr' => 'cdn-jsdelivr',
      'cdnjs' => 'cdn-cdnjs',
      'unpkg' => 'cdn-unpkg',
    ];

    $library_name = $library_map[$provider] ?? 'cdn-jsdelivr';

    // Parse libraries.yml to get the CDN URLs.
    $libraries_file = $this->appRoot . '/' . $this->moduleExtensionList->getPath('lightgallery_formatter') . '/lightgallery_formatter.libraries.yml';
    if (file_exists($libraries_file)) {
      $libraries = yaml_parse(file_get_contents($libraries_file));

      if (isset($libraries[$library_name])) {
        $library = $libraries[$library_name];

        // Extract JS URL.
        $js_url = '';
        if (!empty($library['js'])) {
          foreach (array_keys($library['js']) as $js_path) {
            // Skip options and only get the URL.
            if (is_string($js_path) && strpos($js_path, 'http') === 0) {
              $js_url = $js_path;
              break;
            }
          }
        }

        // Extract CSS URL.
        $css_url = '';
        if (!empty($library['css']['component'])) {
          foreach (array_keys($library['css']['component']) as $css_path) {
            // Skip options and only get the URL.
            if (is_string($css_path) && strpos($css_path, 'http') === 0) {
              $css_url = $css_path;
              break;
            }
          }
        }

        return [
          'js' => $js_url,
          'css' => $css_url,
        ];
      }
    }

    // Fallback to hardcoded values if libraries file cannot be parsed.
    $version = '2.9.0';
    switch ($provider) {
      case 'cdnjs':
        return [
          'js' => "https://cdnjs.cloudflare.com/ajax/libs/lightgallery/{$version}/lightgallery.umd.min.js",
          'css' => "https://cdnjs.cloudflare.com/ajax/libs/lightgallery/{$version}/css/lightgallery-bundle.min.css",
        ];

      case 'unpkg':
        return [
          'js' => "https://unpkg.com/lightgallery@{$version}/lightgallery.umd.min.js",
          'css' => "https://unpkg.com/lightgallery@{$version}/css/lightgallery-bundle.min.css",
        ];

      case 'jsdelivr':
      default:
        return [
          'js' => "https://cdn.jsdelivr.net/npm/lightgallery@{$version}/lightgallery.umd.min.js",
          'css' => "https://cdn.jsdelivr.net/npm/lightgallery@{$version}/css/lightgallery-bundle.min.css",
        ];
    }
  }

  /**
   * AJAX callback to update CDN URLs display.
   */
  public function updateCdnUrls(array &$form, FormStateInterface $form_state) {
    return $form['cdn_urls'];
  }

  /**
   * AJAX callback to generate the gallery preview.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The preview content render array.
   */
  public function generatePreview(array &$form, FormStateInterface $form_state) {
    // Set flag to show preview.
    $form_state->set('show_preview', TRUE);

    // Rebuild the preview content.
    $form['preview']['content']['gallery'] = $this->buildPreviewGallery($form_state);

    // Remove placeholder if exists.
    unset($form['preview']['content']['placeholder']);

    return $form['preview']['content'];
  }

  /**
   * Build the preview gallery render array.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The gallery render array.
   */
  protected function buildPreviewGallery(FormStateInterface $form_state) {
    // Check if preview module is enabled.
    if (!$this->moduleHandler->moduleExists('lightgallery_formatter_preview')) {
      return [
        '#type' => 'container',
        '#attributes' => ['class' => ['lightgallery-preview-not-available']],
        'message' => [
          '#markup' => '<div class="messages messages--warning">' .
          $this->t('The LightGallery Formatter Preview module is not enabled. Please enable it to use the preview functionality.') .
          '</div>',
        ],
      ];
    }

    // Build settings from form state.
    $settings = $this->buildLightgallerySettingsFromForm($form_state);

    // Use preview builder service.
    if ($this->previewBuilder) {
      return $this->previewBuilder->buildPreview($settings);
    }

    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['lightgallery-preview-error']],
      'message' => [
        '#markup' => '<div class="messages messages--error">' .
        $this->t('Preview service is not available.') .
        '</div>',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    /** @var \Drupal\lightgallery_formatter\Entity\LightgalleryProfile $profile */
    $profile = $this->entity;

    // Save general settings.
    $general_settings = [
      'loop' => (bool) $form_state->getValue('loop'),
      'preload' => (int) $form_state->getValue('preload'),
      'speed' => (int) $form_state->getValue('speed'),
      'mode' => $form_state->getValue('mode'),
      'closable' => (bool) $form_state->getValue('closable'),
      'closeOnTap' => (bool) $form_state->getValue('closeOnTap'),
      'escKey' => (bool) $form_state->getValue('escKey'),
      'keyPress' => (bool) $form_state->getValue('keyPress'),
      'showCloseIcon' => (bool) $form_state->getValue('showCloseIcon'),
      'showMaximizeIcon' => (bool) $form_state->getValue('showMaximizeIcon'),
      'hideBarsDelay' => (int) $form_state->getValue('hideBarsDelay'),
      'counter' => (bool) $form_state->getValue('counter'),
      'download' => (bool) $form_state->getValue('download'),
      'enableDrag' => (bool) $form_state->getValue('enableDrag'),
      'enableSwipe' => (bool) $form_state->getValue('enableSwipe'),
      'mousewheel' => (bool) $form_state->getValue('mousewheel'),
    ];
    $profile->setPluginSettings('general', $general_settings);

    // Save thumbnail settings if enabled.
    if ($form_state->getValue('enable_thumbnail')) {
      $thumbnail_settings = [
        'thumbnail' => TRUE,
        'animateThumb' => (bool) $form_state->getValue('animateThumb'),
        'showThumbByDefault' => (bool) $form_state->getValue('showThumbByDefault'),
        'image_style' => $form_state->getValue('image_style') ?? '',
        'thumbMargin' => (int) $form_state->getValue('thumbMargin'),
        'appendTo' => $form_state->getValue('appendTo') ?? '',
      ];
      $profile->setPluginSettings('thumbnail', $thumbnail_settings);
    }
    else {
      // Clear thumbnail settings if disabled.
      $profile->setPluginSettings('thumbnail', []);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $profile = $this->entity;
    $status = $profile->save();

    if ($status === SAVED_NEW) {
      $this->messenger()->addStatus($this->t('Created the %label LightGallery profile.', [
        '%label' => $profile->label(),
      ]));
    }
    else {
      $this->messenger()->addStatus($this->t('Saved the %label LightGallery profile.', [
        '%label' => $profile->label(),
      ]));
    }

    $form_state->setRedirectUrl($profile->toUrl('collection'));

    return $status;
  }

  /**
   * Build LightGallery settings from form state.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   LightGallery settings array.
   */
  protected function buildLightgallerySettingsFromForm(FormStateInterface $form_state) {
    $profile = $this->entity;

    // Get general settings from form or entity.
    $general_settings = $profile->getValidatedPluginSettings('general');
    $loop = $form_state->getValue('loop') ?? $general_settings['loop'];
    $speed = $form_state->getValue('speed') ?? $general_settings['speed'];
    $mode = $form_state->getValue('mode') ?? $general_settings['mode'];
    $closable = $form_state->getValue('closable') ?? $general_settings['closable'];
    $closeOnTap = $form_state->getValue('closeOnTap') ?? $general_settings['closeOnTap'];
    $escKey = $form_state->getValue('escKey') ?? $general_settings['escKey'];
    $keyPress = $form_state->getValue('keyPress') ?? $general_settings['keyPress'];
    $showCloseIcon = $form_state->getValue('showCloseIcon') ?? $general_settings['showCloseIcon'];
    $showMaximizeIcon = $form_state->getValue('showMaximizeIcon') ?? $general_settings['showMaximizeIcon'];
    $hideBarsDelay = $form_state->getValue('hideBarsDelay') ?? $general_settings['hideBarsDelay'];
    $counter = $form_state->getValue('counter') ?? $general_settings['counter'];
    $download = $form_state->getValue('download') ?? $general_settings['download'];
    $enableDrag = $form_state->getValue('enableDrag') ?? $general_settings['enableDrag'];
    $enableSwipe = $form_state->getValue('enableSwipe') ?? $general_settings['enableSwipe'];
    $mousewheel = $form_state->getValue('mousewheel') ?? $general_settings['mousewheel'];
    $preload = $form_state->getValue('preload') ?? $general_settings['preload'];

    // Get thumbnail settings.
    $enable_thumbnail = $form_state->getValue('enable_thumbnail') ?? !empty($profile->getPluginSettings('thumbnail'));
    $thumbnail_settings = $profile->getValidatedPluginSettings('thumbnail');
    $animateThumb = $form_state->getValue('animateThumb') ?? $thumbnail_settings['animateThumb'] ?? TRUE;
    $showThumbByDefault = $form_state->getValue('showThumbByDefault') ?? $thumbnail_settings['showThumbByDefault'] ?? TRUE;
    $thumbMargin = $form_state->getValue('thumbMargin') ?? $thumbnail_settings['thumbMargin'] ?? 5;

    // Build LightGallery settings for JS.
    $settings = [
      'selector' => 'a',
      'loop' => (bool) $loop,
      'speed' => (int) $speed,
      'mode' => $mode,
      'closable' => (bool) $closable,
      'closeOnTap' => (bool) $closeOnTap,
      'escKey' => (bool) $escKey,
      'keyPress' => (bool) $keyPress,
      'showCloseIcon' => (bool) $showCloseIcon,
      'showMaximizeIcon' => (bool) $showMaximizeIcon,
      'hideBarsDelay' => (int) $hideBarsDelay,
      'counter' => (bool) $counter,
      'download' => (bool) $download,
      'enableDrag' => (bool) $enableDrag,
      'enableSwipe' => (bool) $enableSwipe,
      'mousewheel' => (bool) $mousewheel,
      'preload' => (int) $preload,
    ];

    // Add thumbnail settings if enabled.
    if ($enable_thumbnail) {
      $settings['enableThumbnail'] = TRUE;
      $settings['thumbnail'] = TRUE;
      $settings['animateThumb'] = (bool) $animateThumb;
      $settings['showThumbByDefault'] = (bool) $showThumbByDefault;
      $settings['thumbMargin'] = (int) $thumbMargin;
    }

    return $settings;
  }

}
