<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\user\Entity\Role;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\user\Entity\User;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;

/**
 * Tests CRUD operations for LightgalleryProfile configuration entities.
 *
 * @group lightgallery_formatter
 */
class LightgalleryProfileCRUDTest extends EntityKernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter', 'user'];

  /**
   * The entity storage handler.
   *
   * @var \Drupal\Core\Config\Entity\ConfigEntityStorageInterface
   */
  protected $storage;

  /**
   * A user with administrative permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
    // Create a regular user.
    $user = User::create([
      'name' => 'authorized_user',
      'mail' => 'authorized@example.com',
      'status' => 1,
    ]);
    $user->save();

    // Create a custom role with only the required permission.
    $role = Role::create([
      'id' => 'lightgallery_admin',
      'label' => 'Lightgallery Administrator',
    ]);
    $role->grantPermission('administer lightgallery_formatter');
    $role->save();

    // Assign the custom role to the user.
    $user->addRole('lightgallery_admin');
    $user->save();
    $this->adminUser = $user;

    // Set up storage.
    $this->storage = $this->container->get('entity_type.manager')->getStorage('lightgallery_profile');

  }

  /**
   * Test creating a LightgalleryProfile entity.
   */
  public function testCreate() {
    $this->assertTrue($this->adminUser->hasPermission('administer lightgallery_formatter'), 'Admin user has required permission.');

    // Create a new profile.
    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
      'status' => TRUE,
    ]);

    // Set thumbnail plugin settings.
    $profile->setPluginSettings('thumbnail', [
      'thumbnail' => TRUE,
      'animateThumb' => FALSE,
      'image_style' => 'thumbnail',
      'thumbMargin' => 5,
      'appendTo' => '',
      'showThumbByDefault' => TRUE,
    ]);

    // Set general plugin settings with full configuration.
    $profile->setPluginSettings('general', [
      'loop' => FALSE,
      'preload' => 3,
      'speed' => 800,
      'mode' => 'lg-zoom-in',
      'closable' => TRUE,
      'closeOnTap' => FALSE,
      'escKey' => TRUE,
      'keyPress' => TRUE,
      'showCloseIcon' => TRUE,
      'showMaximizeIcon' => FALSE,
      'hideBarsDelay' => 1500,
      'counter' => TRUE,
      'download' => FALSE,
      'enableDrag' => TRUE,
      'enableSwipe' => TRUE,
      'mousewheel' => FALSE,
    ]);

    // Save it.
    $profile->save();

    // Reload from storage.
    $this->storage->resetCache();
    $reloaded = $this->storage->load('test_profile');

    // Assert values.
    $this->assertNotNull($reloaded, 'Profile was saved and can be loaded.');
    $this->assertEquals('Test Profile', $reloaded->label(), 'Label is correct.');
    $this->assertTrue($reloaded->getStatus(), 'Status is enabled.');

    // Check thumbnail settings.
    $thumbnail_settings = $reloaded->getPluginSettings('thumbnail');
    $this->assertTrue($thumbnail_settings['thumbnail'], 'Thumbnail is enabled.');
    $this->assertEquals('thumbnail', $thumbnail_settings['image_style'], 'Thumbnail image style is correct.');
    $this->assertFalse($thumbnail_settings['animateThumb'], 'Thumbnail animation is disabled.');

    // Check general settings.
    $general_settings = $reloaded->getPluginSettings('general');
    $this->assertFalse($general_settings['loop'], 'Loop is disabled.');
    $this->assertEquals(3, $general_settings['preload'], 'Preload count is correct.');
    $this->assertEquals(800, $general_settings['speed'], 'Transition speed is correct.');
    $this->assertEquals('lg-zoom-in', $general_settings['mode'], 'Transition mode is correct.');
    $this->assertTrue($general_settings['closable'], 'Closable setting is correct.');
    $this->assertFalse($general_settings['closeOnTap'], 'Close on tap setting is correct.');
    $this->assertTrue($general_settings['escKey'], 'Esc key setting is correct.');
    $this->assertTrue($general_settings['keyPress'], 'Keyboard navigation setting is correct.');
    $this->assertTrue($general_settings['showCloseIcon'], 'Show close icon setting is correct.');
    $this->assertFalse($general_settings['showMaximizeIcon'], 'Show maximize icon setting is correct.');
    $this->assertEquals(1500, $general_settings['hideBarsDelay'], 'Hide bars delay is correct.');
    $this->assertTrue($general_settings['counter'], 'Counter setting is correct.');
    $this->assertFalse($general_settings['download'], 'Download setting is correct.');
    $this->assertTrue($general_settings['enableDrag'], 'Enable drag setting is correct.');
    $this->assertTrue($general_settings['enableSwipe'], 'Enable swipe setting is correct.');
    $this->assertFalse($general_settings['mousewheel'], 'Mouse wheel setting is correct.');

    // Test convenience methods.
    $this->assertFalse($reloaded->getLoop(), 'Loop convenience method returns correct value.');
    $this->assertEquals(3, $reloaded->getPreload(), 'Preload convenience method returns correct value.');
  }

  /**
   * Test reading an existing LightgalleryProfile entity.
   */
  public function testRead() {
    // Manually add config (simulate existing).
    $this->container->get('config.factory')->getEditable('lightgallery_formatter.profile.test_profile')
      ->setData([
        'langcode' => 'en',
        'status' => TRUE,
        'dependencies' => [],
        'id' => 'test_profile',
        'label' => 'Test Profile',
        'plugin_settings' => [
          'thumbnail' => [
            'thumbnail' => FALSE,
            'animateThumb' => TRUE,
            'image_style' => 'medium',
            'thumbMargin' => 5,
            'appendTo' => '',
            'showThumbByDefault' => TRUE,
          ],
          'general' => [
            'loop' => TRUE,
            'preload' => 2,
            'speed' => 400,
            'mode' => 'lg-slide',
            'closable' => FALSE,
            'closeOnTap' => TRUE,
            'escKey' => FALSE,
            'keyPress' => FALSE,
            'showCloseIcon' => FALSE,
            'showMaximizeIcon' => TRUE,
            'hideBarsDelay' => 0,
            'counter' => FALSE,
            'download' => TRUE,
            'enableDrag' => FALSE,
            'enableSwipe' => FALSE,
            'mousewheel' => TRUE,
          ],
        ],
      ])
      ->save();

    // Load via storage.
    $profile = $this->storage->load('test_profile');
    $this->assertNotNull($profile, 'Profile exists in configuration.');
    $this->assertEquals('Test Profile', $profile->label());

    $thumbnail_settings = $profile->getPluginSettings('thumbnail');
    $this->assertFalse($thumbnail_settings['thumbnail']);
    $this->assertTrue($thumbnail_settings['animateThumb']);
    $this->assertEquals('medium', $thumbnail_settings['image_style']);

    $general_settings = $profile->getPluginSettings('general');
    $this->assertTrue($general_settings['loop']);
    $this->assertEquals(2, $general_settings['preload']);
    $this->assertEquals(400, $general_settings['speed']);
    $this->assertEquals('lg-slide', $general_settings['mode']);
    $this->assertFalse($general_settings['closable']);
    $this->assertTrue($general_settings['closeOnTap']);
    $this->assertFalse($general_settings['escKey']);
    $this->assertFalse($general_settings['keyPress']);
    $this->assertFalse($general_settings['showCloseIcon']);
    $this->assertTrue($general_settings['showMaximizeIcon']);
    $this->assertEquals(0, $general_settings['hideBarsDelay']);
    $this->assertFalse($general_settings['counter']);
    $this->assertTrue($general_settings['download']);
    $this->assertFalse($general_settings['enableDrag']);
    $this->assertFalse($general_settings['enableSwipe']);
    $this->assertTrue($general_settings['mousewheel']);

    // Test convenience methods.
    $this->assertTrue($profile->getLoop(), 'Loop convenience method returns correct value.');
    $this->assertEquals(2, $profile->getPreload(), 'Preload convenience method returns correct value.');
  }

  /**
   * Test updating a LightgalleryProfile entity.
   */
  public function testUpdate() {
    // Create and save initial profile.
    $profile = LightgalleryProfile::create([
      'id' => 'update_test',
      'label' => 'Update Me',
    ]);
    $profile->setPluginSettings('thumbnail', [
      'thumbnail' => FALSE,
      'animateThumb' => TRUE,
      'image_style' => 'thumbnail',
    ]);
    $profile->setPluginSettings('general', [
      'loop' => TRUE,
      'preload' => 2,
      'speed' => 400,
      'mode' => 'lg-slide',
      'closable' => TRUE,
      'closeOnTap' => TRUE,
      'escKey' => TRUE,
      'keyPress' => TRUE,
      'showCloseIcon' => TRUE,
      'showMaximizeIcon' => FALSE,
      'hideBarsDelay' => 0,
      'counter' => TRUE,
      'download' => TRUE,
      'enableDrag' => TRUE,
      'enableSwipe' => TRUE,
      'mousewheel' => FALSE,
    ]);
    $profile->save();

    // Load and modify.
    $profile = $this->storage->load('update_test');
    $this->assertEquals('Update Me', $profile->label());

    $profile->set('label', 'Updated Label');
    $profile->setPluginSettings('thumbnail', [
      'thumbnail' => TRUE,
      'animateThumb' => FALSE,
      'image_style' => 'large',
    ]);
    $profile->setPluginSettings('general', [
      'loop' => FALSE,
      'preload' => 5,
      'speed' => 1000,
      'mode' => 'lg-fade',
      'closable' => FALSE,
      'closeOnTap' => FALSE,
      'escKey' => FALSE,
      'keyPress' => FALSE,
      'showCloseIcon' => FALSE,
      'showMaximizeIcon' => TRUE,
      'hideBarsDelay' => 5000,
      'counter' => FALSE,
      'download' => FALSE,
      'enableDrag' => FALSE,
      'enableSwipe' => FALSE,
      'mousewheel' => TRUE,
    ]);
    $profile->save();

    // Re-fetch.
    $this->storage->resetCache();
    $updated = $this->storage->load('update_test');

    $this->assertEquals('Updated Label', $updated->label(), 'Label was updated.');

    $thumbnail_settings = $updated->getPluginSettings('thumbnail');
    $this->assertTrue($thumbnail_settings['thumbnail'], 'Thumbnail setting was updated.');
    $this->assertEquals('large', $thumbnail_settings['image_style'], 'Thumbnail image style was updated.');

    $general_settings = $updated->getPluginSettings('general');
    $this->assertFalse($general_settings['loop'], 'Loop setting was updated.');
    $this->assertEquals(5, $general_settings['preload'], 'Preload setting was updated.');
    $this->assertEquals(1000, $general_settings['speed'], 'Speed setting was updated.');
    $this->assertEquals('lg-fade', $general_settings['mode'], 'Mode setting was updated.');
    $this->assertFalse($general_settings['closable'], 'Closable setting was updated.');
    $this->assertFalse($general_settings['closeOnTap'], 'Close on tap setting was updated.');
    $this->assertFalse($general_settings['escKey'], 'Esc key setting was updated.');
    $this->assertFalse($general_settings['keyPress'], 'Key press setting was updated.');
    $this->assertFalse($general_settings['showCloseIcon'], 'Show close icon setting was updated.');
    $this->assertTrue($general_settings['showMaximizeIcon'], 'Show maximize icon setting was updated.');
    $this->assertEquals(5000, $general_settings['hideBarsDelay'], 'Hide bars delay setting was updated.');
    $this->assertFalse($general_settings['counter'], 'Counter setting was updated.');
    $this->assertFalse($general_settings['download'], 'Download setting was updated.');
    $this->assertFalse($general_settings['enableDrag'], 'Enable drag setting was updated.');
    $this->assertFalse($general_settings['enableSwipe'], 'Enable swipe setting was updated.');
    $this->assertTrue($general_settings['mousewheel'], 'Mouse wheel setting was updated.');
  }

  /**
   * Test deleting a LightgalleryProfile entity.
   */
  public function testDelete() {
    // Create and save.
    LightgalleryProfile::create([
      'id' => 'delete_me',
      'label' => 'Delete Me',
    ])->save();

    // Confirm exists.
    $profile = $this->storage->load('delete_me');
    $this->assertNotNull($profile, 'Profile exists before deletion.');

    // Delete.
    $profile->delete();

    // Confirm gone.
    $this->storage->resetCache();
    $deleted = $this->storage->load('delete_me');
    $this->assertNull($deleted, 'Profile was deleted and no longer exists.');
  }

  /**
   * Test CDN settings functionality.
   */
  public function testCdnSettings() {
    // Create a new profile with CDN settings.
    $profile = LightgalleryProfile::create([
      'id' => 'cdn_test',
      'label' => 'CDN Test Profile',
      'status' => TRUE,
      'use_cdn' => TRUE,
      'cdn_provider' => 'cdnjs',
    ]);
    $profile->save();

    // Reload from storage.
    $this->storage->resetCache();
    $reloaded = $this->storage->load('cdn_test');

    // Assert CDN values.
    $this->assertNotNull($reloaded, 'Profile was saved and can be loaded.');
    $this->assertEquals('CDN Test Profile', $reloaded->label(), 'Label is correct.');
    $this->assertTrue($reloaded->useCdn(), 'CDN usage is enabled.');
    $this->assertEquals('cdnjs', $reloaded->getCdnProvider(), 'CDN provider is correct.');

    // Test with different CDN provider.
    $profile2 = LightgalleryProfile::create([
      'id' => 'cdn_test2',
      'label' => 'CDN Test Profile 2',
      'status' => FALSE,
      'use_cdn' => TRUE,
      'cdn_provider' => 'unpkg',
    ]);
    $profile2->save();

    // Reload from storage.
    $this->storage->resetCache();
    $reloaded2 = $this->storage->load('cdn_test2');

    // Assert CDN values.
    $this->assertFalse($reloaded2->getStatus(), 'Status is disabled.');
    $this->assertTrue($reloaded2->useCdn(), 'CDN usage is enabled.');
    $this->assertEquals('unpkg', $reloaded2->getCdnProvider(), 'CDN provider is correct.');

    // Test with CDN disabled.
    $profile3 = LightgalleryProfile::create([
      'id' => 'cdn_test3',
      'label' => 'CDN Test Profile 3',
      'use_cdn' => FALSE,
    // This should be ignored when use_cdn is FALSE.
      'cdn_provider' => 'jsdelivr',
    ]);
    $profile3->save();

    // Reload from storage.
    $this->storage->resetCache();
    $reloaded3 = $this->storage->load('cdn_test3');

    // Assert CDN values.
    $this->assertFalse($reloaded3->useCdn(), 'CDN usage is disabled.');
    $this->assertEquals('jsdelivr', $reloaded3->getCdnProvider(), 'CDN provider is stored even when CDN is disabled.');
  }

}
