<?php

namespace Drupal\lightgallery_formatter\Entity;

use Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryGeneralSettings;
use Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryThumbnailSettings;
use Drupal\Core\Config\Entity\ConfigEntityBase;

/**
 * Defines the LightGallery profile configuration entity.
 *
 * @ConfigEntityType(
 *   id = "lightgallery_profile",
 *   label = @Translation("LightGallery profile"),
 *   handlers = {
 *     "list_builder" = "Drupal\lightgallery_formatter\Controller\LightgalleryProfileController",
 *     "form" = {
 *       "add" = "Drupal\lightgallery_formatter\Form\LightgalleryProfileForm",
 *       "edit" = "Drupal\lightgallery_formatter\Form\LightgalleryProfileForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm"
 *     }
 *   },
 *   config_prefix = "profile",
 *   admin_permission = "administer lightgallery_formatter",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "status" = "status"
 *   },
 *   links = {
 *     "collection" = "/admin/structure/lightgallery-profiles",
 *     "add-form" = "/admin/structure/lightgallery-profiles/add",
 *     "edit-form" = "/admin/structure/lightgallery-profiles/{lightgallery_profile}",
 *     "delete-form" = "/admin/structure/lightgallery-profiles/{lightgallery_profile}/delete"
 *   },
 *   config_export = {
 *     "id" = "id",
 *     "label" = "label",
 *     "status" = "status",
 *     "use_cdn" = "use_cdn",
 *     "cdn_provider" = "cdn_provider",
 *     "plugin_settings" = "plugin_settings"
 *   }
 * )
 */
class LightgalleryProfile extends ConfigEntityBase implements LightgalleryProfileInterface {

  /**
   * The machine name of the profile.
   *
   * @var string
   */
  protected $id;

  /**
   * The human-readable name of the profile.
   *
   * @var string
   */
  protected $label;

  /**
   * Whether the profile is enabled.
   *
   * @var bool
   */
  protected $status = TRUE;

  /**
   * Whether to use CDN for loading assets.
   *
   * @var bool
   */
  protected $use_cdn = FALSE;

  /**
   * CDN provider to use.
   *
   * @var string
   */
  protected $cdn_provider = 'jsdelivr';

  /**
   * Plugin configurations.
   *
   * @var array
   */
  protected $plugin_settings = [];

  /**
   * Gets the status.
   */
  public function getStatus(): bool {
    return (bool) $this->status;
  }

  /**
   * Gets whether to use CDN.
   */
  public function useCdn(): bool {
    return (bool) $this->use_cdn;
  }

  /**
   * Gets the CDN provider.
   */
  public function getCdnProvider(): string {
    return $this->cdn_provider;
  }

  /**
   * Get plugin settings.
   */
  public function getPluginSettings(string $plugin_id): array {
    return $this->plugin_settings[$plugin_id] ?? [];
  }

  /**
   * Set plugin settings.
   */
  public function setPluginSettings(string $plugin_id, array $settings): void {
    $this->plugin_settings[$plugin_id] = $settings;
  }

  /**
   * Get all plugin settings.
   */
  public function getAllPluginSettings(): array {
    return $this->plugin_settings;
  }

  /**
   * Get validated plugin settings with defaults.
   */
  public function getValidatedPluginSettings(string $plugin_id): array {
    $plugin_settings = $this->getPluginSettings($plugin_id);

    // Load appropriate plugin class.
    switch ($plugin_id) {
      case 'thumbnail':
        $plugin = new LightgalleryThumbnailSettings();
        break;

      case 'general':
        $plugin = new LightgalleryGeneralSettings();
        break;

      default:
        return $plugin_settings;
    }

    // Merge with defaults and validate.
    $defaults = $plugin->getDefaultSettings();
    $merged_settings = array_merge($defaults, $plugin_settings);

    if ($plugin->validateSettings($merged_settings)) {
      return $merged_settings;
    }

    return $defaults;
  }

  /**
   * Gets whether looping is enabled.
   */
  public function getLoop(): bool {
    $general_settings = $this->getValidatedPluginSettings('general');
    return (bool) ($general_settings['loop'] ?? TRUE);
  }

  /**
   * Gets number of images to preload.
   */
  public function getPreload(): int {
    $general_settings = $this->getValidatedPluginSettings('general');
    return (int) ($general_settings['preload'] ?? 2);
  }

}
