<?php

namespace Drupal\lightgallery_formatter\Plugin\Field\FieldFormatter;

use Drupal\Component\Utility\Html;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Drupal\image\Entity\ImageStyle;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'lightgallery' formatter.
 *
 * @FieldFormatter(
 *   id = "lightgallery_formatter",
 *   label = @Translation("LightGallery"),
 *   field_types = {
 *     "entity_reference"
 *   },
 *   quickedit = {
 *     "editor" = "disabled"
 *   }
 * )
 */
class LightGalleryFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    // @phpstan-ignore-next-line new.static is standard Drupal DI pattern.
    $instance = new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings']
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->currentUser = $container->get('current_user');
    $instance->fileUrlGenerator = $container->get('file_url_generator');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    // Apply only to media image references.
    $is_media_image = $field_definition->getFieldStorageDefinition()->getSetting('target_type') === 'media' &&
      $field_definition->getSetting('handler_settings')['target_bundles']['image'] ?? FALSE;

    return $is_media_image && parent::isApplicable($field_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $profile_id = $this->getSetting('profile');
    $profile = $this->entityTypeManager
      ->getStorage('lightgallery_profile')
      ->load($profile_id);

    if ($profile) {
      $summary[] = $this->t('Profile: @label', ['@label' => $profile->label()]);
    }
    else {
      $summary[] = $this->t('No profile selected');
    }

    // Display per-bundle gallery item source field settings.
    $gallery_item_source_fields = $this->getSetting('gallery_item_source_fields') ?: [];

    if (!empty($gallery_item_source_fields)) {
      $bundle_storage = $this->entityTypeManager->getStorage('media_type');
      foreach ($gallery_item_source_fields as $bundle => $field_name) {
        $bundle_entity = $bundle_storage->load($bundle);
        $bundle_label = $bundle_entity ? $bundle_entity->label() : ucfirst(str_replace('_', ' ', $bundle));

        if ($field_name === '_thumbnail_uri') {
          $summary[] = $this->t('@bundle: Thumbnail (from metadata)', ['@bundle' => $bundle_label]);
        }
        else {
          $summary[] = $this->t('@bundle: @field', ['@bundle' => $bundle_label, '@field' => $field_name]);
        }
      }
    }

    $thumbnail_style_id = $this->getSetting('thumbnail_style');
    if ($thumbnail_style_id) {
      $thumbnail_style = ImageStyle::load($thumbnail_style_id);
      if ($thumbnail_style) {
        $summary[] = $this->t('Thumbnail style: @style', ['@style' => $thumbnail_style->label()]);
      }
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $profiles = $this->entityTypeManager
      ->getStorage('lightgallery_profile')
      ->loadByProperties(['status' => TRUE]);
    $options = [];
    foreach ($profiles as $profile_id => $profile) {
      /** @var \Drupal\Core\Config\Entity\ConfigEntityInterface $profile */
      $options[$profile_id] = $profile->label();
    }

    $profile_list_url = Url::fromRoute('entity.lightgallery_profile.collection');
    $profile_list_link = Link::fromTextAndUrl($this->t('Manage profiles'), $profile_list_url)->toString();

    $element['profile'] = [
      '#type' => 'select',
      '#title' => $this->t('LightGallery Profile'),
      '#description' => $this->t('Choose the display configuration for this gallery. @link', ['@link' => $profile_list_link]),
      '#default_value' => $this->getSetting('profile'),
      '#empty_option' => $this->t('Select a profile...'),
      '#required' => TRUE,
      '#options' => $options,
    ];

    // Add per-bundle gallery item source field selection
    // for entity reference fields.
    if ($this->fieldDefinition->getType() === 'entity_reference') {
      $target_type = $this->fieldDefinition->getSetting('target_type');
      $handler_settings = $this->fieldDefinition->getSetting('handler_settings');
      $target_bundles = $handler_settings['target_bundles'] ?? [];

      if ($target_type === 'media' && !empty($target_bundles)) {
        // Get current settings.
        $gallery_item_source_fields = $this->getSetting('gallery_item_source_fields') ?: [];

        $bundle_storage = $this->entityTypeManager->getStorage('media_type');

        $element['gallery_item_source_fields'] = [
          '#type' => 'details',
          '#title' => $this->t('Gallery item source by media type'),
          '#description' => $this->t('Configure which field to use as the gallery item source for each media type.'),
          '#open' => FALSE,
        ];

        foreach ($target_bundles as $bundle) {
          $bundle_entity = $bundle_storage->load($bundle);
          $bundle_label = $bundle_entity ? $bundle_entity->label() : ucfirst(str_replace('_', ' ', $bundle));
          $is_remote_video = in_array($bundle, ['remote_video', 'video', 'oembed_video']);

          if ($is_remote_video) {
            // For remote video bundles, show readonly info and store the value.
            $element['gallery_item_source_fields'][$bundle] = [
              '#type' => 'textfield',
              '#title' => $bundle_label,
              '#default_value' => '_thumbnail_uri',
              '#attributes' => ['readonly' => 'readonly', 'style' => 'background-color: #f5f5f5;'],
              '#description' => $this->t('Remote video thumbnails are automatically extracted from video metadata (YouTube/Vimeo).'),
            ];
          }
          else {
            // For image bundles, show field selection dropdown.
            $bundle_field_definitions = $this->entityFieldManager->getFieldDefinitions($target_type, $bundle);
            $bundle_fields = [];

            foreach ($bundle_field_definitions as $field_name => $field_definition) {
              // Skip the base 'thumbnail' field - it's for internal use only.
              if ($field_name === 'thumbnail') {
                continue;
              }

              $field_type = $field_definition->getType();
              $field_label = $field_definition->getLabel();
              $field_display = $field_label . ' (' . $field_name . ')';

              // Direct image fields.
              if ($field_type === 'image') {
                $bundle_fields[$field_name] = $field_display;
              }
              // Media reference fields pointing to image bundles.
              elseif ($field_type === 'entity_reference') {
                $ref_target_type = $field_definition->getSetting('target_type');
                $ref_handler_settings = $field_definition->getSetting('handler_settings');
                $ref_target_bundles = $ref_handler_settings['target_bundles'] ?? [];
                if ($ref_target_type === 'media' && !empty($ref_target_bundles) && in_array('image', $ref_target_bundles)) {
                  $bundle_fields[$field_name] = $field_display;
                }
              }
            }

            if (!empty($bundle_fields)) {
              $element['gallery_item_source_fields'][$bundle] = [
                '#type' => 'select',
                '#title' => $bundle_label,
                '#default_value' => $gallery_item_source_fields[$bundle] ?? ($bundle === 'image' ? 'field_media_image' : ''),
                '#empty_option' => $this->t('- Select a field -'),
                '#options' => $bundle_fields,
                '#description' => $this->t('Select which field from @bundle media entities should be used as the gallery item source.', ['@bundle' => $bundle_label]),
              ];
            }
          }
        }
      }
    }

    // Add image style selection for thumbnails.
    $image_styles = ['' => $this->t('- Original -')];
    foreach ($this->entityTypeManager->getStorage('image_style')->loadMultiple() as $image_style) {
      $image_styles[$image_style->id()] = $image_style->label();
    }

    $element['thumbnail_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Thumbnail image style'),
      '#description' => $this->t('Select the image style to use for gallery thumbnails. The full-size image will be used when clicking on thumbnails.'),
      '#default_value' => $this->getSetting('thumbnail_style'),
      '#options' => $image_styles,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'profile' => '',
      'gallery_item_source_fields' => [],
      'thumbnail_style' => 'thumbnail',
    ] + parent::defaultSettings();
  }

  /**
   * Extracts image data from a media entity based on field type.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   * @param string $field_name
   *   The field name to extract image data from.
   *
   * @return array|null
   *   Array with 'full_uri', 'full_url', 'file_entity', 'alt', 'title' keys,
   *   or NULL if not found.
   */
  protected function getFieldImageData($media, $field_name) {
    $thumbnail_style_id = $this->getSetting('thumbnail_style') ?: 'thumbnail';
    if (!$media) {
      return NULL;
    }

    $bundle = $media->bundle();
    $is_remote_video = in_array($bundle, ['remote_video', 'video', 'oembed_video']);

    // For remote video, if _thumbnail_uri is selected or no field specified,
    // use thumbnail from metadata (YouTube/Vimeo).
    if ($is_remote_video && ($field_name === '_thumbnail_uri' || empty($field_name))) {
      $thumbnail_uri = $media->getSource()->getMetadata($media, 'thumbnail_uri');
      if ($thumbnail_uri && file_exists($thumbnail_uri)) {
        $full_url = $this->fileUrlGenerator->generateAbsoluteString($thumbnail_uri);

        return [
          'full_uri' => $thumbnail_uri,
          'full_url' => $full_url,
          'file_entity' => NULL,
          'alt' => $media->label(),
          'title' => $media->label(),
        ];
      }
      // If thumbnail_uri is not available, return NULL.
      return NULL;
    }

    // For non-remote-video media, check if field exists and is not empty.
    if (!$media->hasField($field_name) || $media->{$field_name}->isEmpty()) {
      return NULL;
    }

    $field_definition = $media->getFieldDefinition($field_name);
    if (!$field_definition) {
      return NULL;
    }

    $field_type = $field_definition->getType();
    $field_item = $media->{$field_name}->first();

    // Direct image field.
    if ($field_type === 'image' && $field_item) {
      $file_entity = $field_item->entity;
      if (!$file_entity) {
        return NULL;
      }
      $full_uri = $file_entity->getFileUri();
      $full_url = $this->fileUrlGenerator->generateAbsoluteString($full_uri);
      $alt = $field_item->getValue()['alt'] ?? '';

      return [
        'full_uri' => $full_uri,
        'full_url' => $full_url,
        'file_entity' => $file_entity,
        'alt' => $alt,
        'title' => $media->label(),
      ];
    }

    // Media reference field pointing to image bundles.
    if ($field_type === 'entity_reference' && $field_item) {
      $referenced_media = $field_item->entity;
      if ($referenced_media && $referenced_media->getEntityTypeId() === 'media') {
        // Try to get image from referenced media entity.
        // Default to field_media_image if no specific field is set.
        $image_field_name = 'field_media_image';
        if ($referenced_media->hasField($image_field_name) && !$referenced_media->{$image_field_name}->isEmpty()) {
          $image_item = $referenced_media->{$image_field_name}->first();
          $file_entity = $image_item->entity;
          if (!$file_entity) {
            return NULL;
          }
          $full_uri = $file_entity->getFileUri();
          $full_url = $this->fileUrlGenerator->generateAbsoluteString($full_uri);
          $alt = $image_item->getValue()['alt'] ?? '';

          return [
            'full_uri' => $full_uri,
            'full_url' => $full_url,
            'file_entity' => $file_entity,
            'alt' => $alt,
            'title' => $media->label(),
          ];
        }
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    $profile_id = $this->getSetting('profile');
    $profile = $this->entityTypeManager
      ->getStorage('lightgallery_profile')
      ->load($profile_id);

    if (!$profile) {
      if ($this->currentUser->hasPermission('administer site configuration')) {
        $elements[0] = [
          '#markup' => $this->t('LightGallery: Selected profile not found. Please check your field settings.'),
          '#prefix' => '<div class="messages messages--error">',
          '#suffix' => '</div>',
        ];
      }
      return $elements;
    }

    $images = [];
    $gallery_item_source_fields = $this->getSetting('gallery_item_source_fields') ?: [];

    foreach ($items as $item) {
      $media = $item->entity;
      if (!$media) {
        continue;
      }

      // Get the field name based on the media bundle.
      $bundle = $media->bundle();

      if (isset($gallery_item_source_fields[$bundle])) {
        $image_field_name = $gallery_item_source_fields[$bundle];
      }
      else {
        // Default fallback based on bundle type.
        $is_remote_video = in_array($bundle, ['remote_video', 'video', 'oembed_video']);
        $image_field_name = $is_remote_video ? '_thumbnail_uri' : 'field_media_image';
      }

      // Use the helper method to extract image data.
      $image_data = $this->getFieldImageData($media, $image_field_name);
      if ($image_data) {
        $images[] = $image_data;
      }
    }

    if (empty($images)) {
      return $elements;
    }

    // Generate unique gallery ID.
    $gallery_id = Html::getUniqueId('lightgallery-' . uniqid() . '-' . $items->getName());

    $thumbnail_style_id = $this->getSetting('thumbnail_style') ?: 'thumbnail';

    // Build render array structure matching LightGallery format
    // using Drupal best practices.
    $gallery_items = [];
    foreach ($images as $index => $image) {
      $link_attributes = [
        'href' => $image['full_url'],
      ];

      // Add data-sub-html attribute if title is available.
      if (!empty($image['title'])) {
        $link_attributes['data-sub-html'] = Html::escape($image['title']);
      }

      // Build thumbnail image using #theme => 'image_style'
      // if we have a file entity and style.
      if ($image['file_entity'] && $thumbnail_style_id) {
        $thumbnail_image = [
          '#theme' => 'image_style',
          '#style_name' => $thumbnail_style_id,
          '#uri' => $image['full_uri'],
          '#alt' => $image['alt'] ?? '',
          '#title' => $image['title'] ?? '',
        ];
      }
      else {
        // Fallback to html_tag for remote video or when no style is set.
        $thumbnail_image = [
          '#type' => 'html_tag',
          '#tag' => 'img',
          '#attributes' => [
            'src' => $image['full_url'],
            'alt' => Html::escape($image['alt'] ?? ''),
          ],
        ];
      }

      $gallery_items[$index] = [
        '#type' => 'html_tag',
        '#tag' => 'a',
        '#attributes' => $link_attributes,
        'image' => $thumbnail_image,
      ];
    }

    $elements[0] = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#attributes' => [
        'id' => $gallery_id,
      ],
    ];

    // Add gallery items as children.
    foreach ($gallery_items as $index => $item) {
      $elements[0][$index] = $item;
    }

    // Determine which libraries to attach.
    $libraries = ['lightgallery_formatter/gallery'];

    // Add LightGallery library based on profile settings.
    if ($profile->useCdn()) {
      $cdn_provider = $profile->getCdnProvider() ?: 'jsdelivr';
      $libraries[] = 'lightgallery_formatter/cdn-' . $cdn_provider;
    }
    else {
      $libraries[] = 'lightgallery_formatter/lightgallery.local';
    }

    // Add thumbnail plugin library if thumbnails are enabled.
    $thumbnail_settings = $profile->getPluginSettings('thumbnail');
    if (!empty($thumbnail_settings)) {
      if ($profile->useCdn()) {
        $cdn_provider = $profile->getCdnProvider() ?: 'jsdelivr';
        $libraries[] = 'lightgallery_formatter/cdn-' . $cdn_provider . '-thumbnail';
      }
      else {
        $libraries[] = 'lightgallery_formatter/lightgallery.local.thumbnail';
      }
    }

    // Build gallery settings from profile.
    $general_settings = $profile->getValidatedPluginSettings('general');
    $gallery_settings = [
      'loop' => $general_settings['loop'] ?? TRUE,
      'speed' => $general_settings['speed'] ?? 400,
      'mode' => $general_settings['mode'] ?? 'lg-slide',
      'closable' => $general_settings['closable'] ?? TRUE,
      'closeOnTap' => $general_settings['closeOnTap'] ?? TRUE,
      'escKey' => $general_settings['escKey'] ?? TRUE,
      'keyPress' => $general_settings['keyPress'] ?? TRUE,
      'showCloseIcon' => $general_settings['showCloseIcon'] ?? TRUE,
      'showMaximizeIcon' => $general_settings['showMaximizeIcon'] ?? FALSE,
      'hideBarsDelay' => $general_settings['hideBarsDelay'] ?? 0,
      'counter' => $general_settings['counter'] ?? TRUE,
      'download' => $general_settings['download'] ?? FALSE,
      'enableDrag' => $general_settings['enableDrag'] ?? TRUE,
      'enableSwipe' => $general_settings['enableSwipe'] ?? TRUE,
      'mousewheel' => $general_settings['mousewheel'] ?? FALSE,
      'preload' => $general_settings['preload'] ?? 2,
    ];

    // Add thumbnail settings if enabled.
    if (!empty($thumbnail_settings)) {
      $gallery_settings['thumbnail'] = TRUE;
      $gallery_settings['animateThumb'] = $thumbnail_settings['animateThumb'] ?? TRUE;
      $gallery_settings['showThumbByDefault'] = $thumbnail_settings['showThumbByDefault'] ?? TRUE;
      $gallery_settings['thumbMargin'] = $thumbnail_settings['thumbMargin'] ?? 5;
    }

    // Attach library and settings.
    $elements[0]['#attached'] = [
      'library' => $libraries,
      'drupalSettings' => [
        'lightgallery_formatter' => [
          'galleries' => [
            $gallery_id => $gallery_settings,
          ],
        ],
      ],
    ];

    return $elements;
  }

}
