<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\user\Entity\Role;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\user\Entity\User;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;
use Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryThumbnailSettings;
use Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryGeneralSettings;

/**
 * Tests plugin settings for LightgalleryProfile configuration entities.
 *
 * @group lightgallery_formatter
 */
class LightgalleryProfilePluginSettingsTest extends EntityKernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter', 'user', 'image'];

  /**
   * The entity storage handler.
   *
   * @var \Drupal\Core\Config\Entity\ConfigEntityStorageInterface
   */
  protected $storage;

  /**
   * A user with administrative permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter', 'image']);

    // Create a regular user.
    $user = User::create([
      'name' => 'authorized_user',
      'mail' => 'authorized@example.com',
      'status' => 1,
    ]);
    $user->save();

    // Create a custom role with only the required permission.
    $role = Role::create([
      'id' => 'lightgallery_admin',
      'label' => 'Lightgallery Administrator',
    ]);
    $role->grantPermission('administer lightgallery_formatter');
    $role->save();

    // Assign the custom role to the user.
    $user->addRole('lightgallery_admin');
    $user->save();
    $this->adminUser = $user;

    // Set up storage.
    $this->storage = $this->container->get('entity_type.manager')->getStorage('lightgallery_profile');

  }

  /**
   * Test plugin settings functionality.
   */
  public function testPluginSettings() {
    // Create a new profile with plugin settings.
    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
      'status' => TRUE,
    ]);

    // Set thumbnail plugin settings.
    $thumbnail_settings = [
      'thumbnail' => TRUE,
      'animateThumb' => FALSE,
      'image_style' => 'thumbnail',
      'thumbMargin' => 10,
      'appendTo' => '.custom-container',
      'showThumbByDefault' => FALSE,
    ];

    $profile->setPluginSettings('thumbnail', $thumbnail_settings);

    // Set general plugin settings with all available options.
    $general_settings = [
      'loop' => FALSE,
      'preload' => 5,
      'speed' => 600,
      'mode' => 'lg-fade',
      'closable' => FALSE,
      'closeOnTap' => FALSE,
      'escKey' => FALSE,
      'keyPress' => FALSE,
      'showCloseIcon' => FALSE,
      'showMaximizeIcon' => TRUE,
      'hideBarsDelay' => 3000,
      'counter' => FALSE,
      'download' => FALSE,
      'enableDrag' => FALSE,
      'enableSwipe' => FALSE,
      'mousewheel' => TRUE,
    ];

    $profile->setPluginSettings('general', $general_settings);
    $profile->save();

    // Reload from storage.
    $this->storage->resetCache();
    $reloaded = $this->storage->load('test_profile');

    // Assert basic values.
    $this->assertNotNull($reloaded, 'Profile was saved and can be loaded.');
    $this->assertEquals('Test Profile', $reloaded->label(), 'Label is correct.');
    $this->assertTrue($reloaded->getStatus(), 'Status is enabled.');

    // Check thumbnail plugin settings.
    $loaded_thumbnail_settings = $reloaded->getPluginSettings('thumbnail');
    $this->assertEquals($thumbnail_settings, $loaded_thumbnail_settings, 'Thumbnail plugin settings are correctly saved and loaded.');

    // Check general plugin settings.
    $loaded_general_settings = $reloaded->getPluginSettings('general');
    $this->assertEquals($general_settings, $loaded_general_settings, 'General plugin settings are correctly saved and loaded.');

    // Verify each general setting value individually.
    foreach ($general_settings as $key => $value) {
      $this->assertEquals($value, $loaded_general_settings[$key], "General setting '{$key}' is correctly saved and loaded.");
    }

    // Test validated settings with defaults.
    $validated_settings = $reloaded->getValidatedPluginSettings('thumbnail');
    $this->assertEquals($thumbnail_settings, $validated_settings, 'Validated settings match original settings.');

    // Test default settings.
    $thumbnail_plugin = new LightgalleryThumbnailSettings();
    $default_settings = $thumbnail_plugin->getDefaultSettings();
    $profile2 = LightgalleryProfile::create([
      'id' => 'test_profile2',
      'label' => 'Test Profile 2',
    ]);

    $default_validated = $profile2->getValidatedPluginSettings('thumbnail');
    $this->assertEquals($default_settings, $default_validated, 'Default settings are correctly applied.');

    // Test convenience methods.
    $this->assertFalse($reloaded->getLoop(), 'Loop convenience method returns correct value.');
    $this->assertEquals(5, $reloaded->getPreload(), 'Preload convenience method returns correct value.');

    // Test disabling plugin by setting empty array.
    $profile2->setPluginSettings('thumbnail', []);
    $disabled_settings = $profile2->getPluginSettings('thumbnail');
    $this->assertEmpty($disabled_settings, 'Plugin can be disabled by setting empty array.');
  }

  /**
   * Test plugin settings validation.
   */
  public function testPluginSettingsValidation() {
    $profile = LightgalleryProfile::create([
      'id' => 'validation_test',
      'label' => 'Validation Test',
    ]);

    // Test with valid settings.
    $valid_settings = [
      'thumbnail' => TRUE,
      'animateThumb' => TRUE,
      'image_style' => 'medium',
      'thumbMargin' => 10,
    ];

    $profile->setPluginSettings('thumbnail', $valid_settings);
    $validated = $profile->getValidatedPluginSettings('thumbnail');
    // Should include defaults for missing values.
    $this->assertTrue(isset($validated['showThumbByDefault']), 'Default value for showThumbByDefault is set.');
    $this->assertTrue(isset($validated['appendTo']), 'Default value for appendTo is set.');

    // Test general settings validation with partial data.
    $general_valid_settings = [
      'loop' => TRUE,
      'preload' => 3,
      'speed' => 500,
      'mode' => 'lg-slide',
    ];

    $profile->setPluginSettings('general', $general_valid_settings);
    $general_validated = $profile->getValidatedPluginSettings('general');

    // Merge expected values with defaults for comparison.
    $general_plugin = new LightgalleryGeneralSettings();
    $default_general_settings = $general_plugin->getDefaultSettings();
    $expected_with_defaults = array_merge($default_general_settings, $general_valid_settings);

    foreach ($expected_with_defaults as $key => $value) {
      $this->assertEquals($value, $general_validated[$key], "Validated general setting '{$key}' matches expected.");
    }
  }

  /**
   * Test that missing general settings use defaults.
   */
  public function testGeneralSettingsDefaults() {
    $profile = LightgalleryProfile::create([
      'id' => 'defaults_test',
      'label' => 'Defaults Test',
    ]);
    $profile->save();

    $validated = $profile->getValidatedPluginSettings('general');
    $defaults = (new LightgalleryGeneralSettings())->getDefaultSettings();

    foreach ($defaults as $key => $value) {
      $this->assertEquals($value, $validated[$key], "Default value for '{$key}' is applied.");
    }
  }

  /**
   * Test CDN settings functionality.
   */
  public function testCdnSettings() {
    $profile = LightgalleryProfile::create([
      'id' => 'cdn_url_test',
      'label' => 'CDN URL Test',
      'use_cdn' => TRUE,
      'cdn_provider' => 'jsdelivr',
    ]);
    $profile->save();

    // Test with different providers.
    $profile->set('cdn_provider', 'cdnjs');
    $profile->save();

    $this->storage->resetCache();
    $reloaded = $this->storage->load('cdn_url_test');
    $this->assertEquals('cdnjs', $reloaded->getCdnProvider(), 'CDN provider can be changed.');

    $profile->set('cdn_provider', 'unpkg');
    $profile->save();

    $this->storage->resetCache();
    $reloaded = $this->storage->load('cdn_url_test');
    $this->assertEquals('unpkg', $reloaded->getCdnProvider(), 'CDN provider can be changed to unpkg.');

    // Test with CDN disabled.
    $profile->set('use_cdn', FALSE);
    $profile->save();

    $this->storage->resetCache();
    $reloaded = $this->storage->load('cdn_url_test');
    $this->assertFalse($reloaded->useCdn(), 'CDN can be disabled.');
  }

}
