<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;

/**
 * Tests routing for LightgalleryProfile entities.
 *
 * @group lightgallery_formatter
 */
class LightgalleryProfileRoutingTest extends EntityKernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter', 'user'];

  /**
   * A user with administrative permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);

    // Create a regular user.
    $user = User::create([
      'name' => 'authorized_user',
      'mail' => 'authorized@example.com',
      'status' => 1,
    ]);
    $user->save();

    // Create a custom role with only the required permission.
    $role = Role::create([
      'id' => 'lightgallery_admin',
      'label' => 'Lightgallery Administrator',
    ]);
    $role->grantPermission('administer lightgallery_formatter');
    $role->save();

    // Assign the custom role to the user.
    $user->addRole('lightgallery_admin');
    $user->save();
    $this->adminUser = $user;
  }

  /**
   * Tests that routes exist and have proper permissions.
   */
  public function testRouteDefinitions() {
    $route_provider = \Drupal::service('router.route_provider');

    // Test that the collection route exists.
    $collection_route = $route_provider->getRouteByName('entity.lightgallery_profile.collection');
    $this->assertEquals('administer lightgallery_formatter', $collection_route->getRequirement('_permission'));

    // Test that the add form route exists.
    $add_route = $route_provider->getRouteByName('entity.lightgallery_profile.add_form');
    $this->assertEquals('administer lightgallery_formatter', $add_route->getRequirement('_permission'));

    // Test that the edit form route exists.
    $edit_route = $route_provider->getRouteByName('entity.lightgallery_profile.edit_form');
    $this->assertEquals('administer lightgallery_formatter', $edit_route->getRequirement('_permission'));

    // Test that the delete form route exists.
    $delete_route = $route_provider->getRouteByName('entity.lightgallery_profile.delete_form');
    $this->assertEquals('administer lightgallery_formatter', $delete_route->getRequirement('_permission'));
  }

  /**
   * Tests that routes are properly protected by permissions.
   */
  public function testRoutePermissions() {
    $account_switcher = \Drupal::service('account_switcher');

    // Create a user without permissions.
    $user = $this->createUser();

    // Switch to user without permissions.
    $account_switcher->switchTo($user);

    // Test access denied for user without permissions.
    $access_manager = \Drupal::service('access_manager');
    $this->assertFalse($access_manager->checkNamedRoute('entity.lightgallery_profile.collection'),
      'User without permissions should not be able to access collection route');
    $this->assertFalse($access_manager->checkNamedRoute('entity.lightgallery_profile.add_form'),
      'User without permissions should not be able to access add route');

    // Create a profile for testing edit and delete routes.
    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
    ]);
    $profile->save();

    $this->assertFalse($access_manager->checkNamedRoute('entity.lightgallery_profile.edit_form', ['lightgallery_profile' => 'test_profile']),
      'User without permissions should not be able to access edit route');
    $this->assertFalse($access_manager->checkNamedRoute('entity.lightgallery_profile.delete_form', ['lightgallery_profile' => 'test_profile']),
      'User without permissions should not be able to access delete route');

    // Switch back.
    $account_switcher->switchBack();

    // Switch to user with permissions.
    $account_switcher->switchTo($this->adminUser);

    // Test access granted for user with permissions.
    $this->assertTrue($access_manager->checkNamedRoute('entity.lightgallery_profile.collection'),
      'User with permissions should be able to access collection route');
    $this->assertTrue($access_manager->checkNamedRoute('entity.lightgallery_profile.add_form'),
      'User with permissions should be able to access add route');
    $this->assertTrue($access_manager->checkNamedRoute('entity.lightgallery_profile.edit_form', ['lightgallery_profile' => 'test_profile']),
      'User with permissions should be able to access edit route');
    $this->assertTrue($access_manager->checkNamedRoute('entity.lightgallery_profile.delete_form', ['lightgallery_profile' => 'test_profile']),
      'User with permissions should be able to access delete route');

    // Switch back.
    $account_switcher->switchBack();
  }

}
