<?php

namespace Drupal\lightgallery_formatter;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\lightgallery_formatter\Annotation\LightgalleryPlugin;

/**
 * Manages discovery and instantiation of LightGallery plugins.
 *
 * LightGallery plugins provide feature-specific configuration for the
 * LightGallery JavaScript library, such as thumbnails, zoom, video, etc.
 *
 * @see \Drupal\lightgallery_formatter\Annotation\LightgalleryPlugin
 * @see \Drupal\lightgallery_formatter\LightgalleryPluginInterface
 * @see \Drupal\lightgallery_formatter\LightgalleryPluginBase
 * @see plugin_api
 */
class LightgalleryPluginManager extends DefaultPluginManager {

  /**
   * Constructs a LightgalleryPluginManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/Lightgallery',
      $namespaces,
      $module_handler,
      LightgalleryPluginInterface::class,
      LightgalleryPlugin::class
    );

    $this->alterInfo('lightgallery_plugin_info');
    $this->setCacheBackend($cache_backend, 'lightgallery_plugins');
  }

  /**
   * Gets all plugin definitions sorted by weight.
   *
   * @return array
   *   An array of plugin definitions sorted by weight.
   */
  public function getSortedDefinitions(): array {
    $definitions = $this->getDefinitions();

    uasort($definitions, function ($a, $b) {
      return ($a['weight'] ?? 0) <=> ($b['weight'] ?? 0);
    });

    return $definitions;
  }

  /**
   * Creates instances of all plugins with given configurations.
   *
   * @param array $plugin_settings
   *   An array of plugin settings keyed by plugin ID.
   *
   * @return \Drupal\lightgallery_formatter\LightgalleryPluginInterface[]
   *   An array of plugin instances keyed by plugin ID.
   */
  public function createInstances(array $plugin_settings = []): array {
    $instances = [];
    foreach ($this->getSortedDefinitions() as $plugin_id => $definition) {
      $configuration = $plugin_settings[$plugin_id] ?? [];
      $instances[$plugin_id] = $this->createInstance($plugin_id, $configuration);
    }
    return $instances;
  }

}
