<?php

namespace Drupal\lightgallery_formatter\Plugin\Lightgallery;

use Drupal\Core\Form\FormStateInterface;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface;
use Drupal\lightgallery_formatter\LightgalleryPluginBase;

/**
 * Provides fullscreen plugin for LightGallery.
 *
 * @LightgalleryPlugin(
 *   id = "fullscreen",
 *   label = @Translation("Fullscreen"),
 *   description = @Translation("Enable fullscreen mode for the gallery."),
 *   weight = 20
 * )
 */
class Fullscreen extends LightgalleryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultSettings(): array {
    return [
      'enabled' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(array $settings): bool {
    return !empty($settings['enabled']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(
    array &$form,
    FormStateInterface $form_state,
    LightgalleryProfileInterface $profile,
  ): array {
    $settings = $this->mergeWithDefaults($profile->getPluginSettings('fullscreen'));

    $element = [
      '#type' => 'details',
      '#title' => $this->t('Fullscreen'),
      '#open' => !empty($settings['enabled']),
    ];

    $element['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable fullscreen plugin'),
      '#default_value' => $settings['enabled'],
      '#description' => $this->t('Allow users to view the gallery in fullscreen mode.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // No validation needed for fullscreen plugin.
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): array {
    $plugin_id = $this->getPluginId();
    $values = $form_state->getValue($plugin_id) ?? [];

    return [
      'enabled' => (bool) ($values['enabled'] ?? FALSE),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildJsSettings(array $settings): array {
    // Only return settings if enabled.
    if (!$this->isEnabled($settings)) {
      return [];
    }

    return [
      'fullScreen' => TRUE,
    ];
  }

}
