<?php

namespace Drupal\lightgallery_formatter\Plugin\Lightgallery;

use Drupal\Core\Form\FormStateInterface;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface;
use Drupal\lightgallery_formatter\LightgalleryPluginBase;

/**
 * Provides share plugin for LightGallery.
 *
 * @LightgalleryPlugin(
 *   id = "share",
 *   label = @Translation("Share"),
 *   description = @Translation("Enable social media sharing options for gallery images."),
 *   weight = 50
 * )
 */
class Share extends LightgalleryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultSettings(): array {
    return [
      'enabled' => FALSE,
      'share' => TRUE,
      'facebook' => TRUE,
      'facebookDropdownText' => 'Facebook',
      'twitter' => TRUE,
      'twitterDropdownText' => 'Twitter',
      'pinterest' => TRUE,
      'pinterestDropdownText' => 'Pinterest',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(array $settings): bool {
    return !empty($settings['enabled']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array &$form, FormStateInterface $form_state, LightgalleryProfileInterface $profile): array {
    $settings = $this->mergeWithDefaults($profile->getPluginSettings('share'));
    $plugin_id = $this->getPluginId();

    $element = [
      '#type' => 'details',
      '#title' => $this->t('Share'),
      '#open' => !empty($settings['enabled']),
    ];

    $element['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable share plugin'),
      '#default_value' => $settings['enabled'],
      '#description' => $this->t('Enable social media sharing options for gallery images.'),
    ];

    $element['settings_wrapper'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="' . $plugin_id . '[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $element['settings_wrapper']['share'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable share button'),
      '#default_value' => $settings['share'],
      '#description' => $this->t('Show the share button in the toolbar.'),
    ];

    // Facebook settings.
    $element['settings_wrapper']['facebook'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Facebook share'),
      '#default_value' => $settings['facebook'],
      '#description' => $this->t('Allow sharing images to Facebook.'),
    ];

    $element['settings_wrapper']['facebookDropdownText'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Facebook button text'),
      '#default_value' => $settings['facebookDropdownText'],
      '#size' => 30,
      '#description' => $this->t('Text displayed for the Facebook share option.'),
      '#states' => [
        'visible' => [
          ':input[name="' . $plugin_id . '[settings_wrapper][facebook]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Twitter settings.
    $element['settings_wrapper']['twitter'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Twitter share'),
      '#default_value' => $settings['twitter'],
      '#description' => $this->t('Allow sharing images to Twitter.'),
    ];

    $element['settings_wrapper']['twitterDropdownText'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Twitter button text'),
      '#default_value' => $settings['twitterDropdownText'],
      '#size' => 30,
      '#description' => $this->t('Text displayed for the Twitter share option.'),
      '#states' => [
        'visible' => [
          ':input[name="' . $plugin_id . '[settings_wrapper][twitter]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Pinterest settings.
    $element['settings_wrapper']['pinterest'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Pinterest share'),
      '#default_value' => $settings['pinterest'],
      '#description' => $this->t('Allow sharing images to Pinterest.'),
    ];

    $element['settings_wrapper']['pinterestDropdownText'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pinterest button text'),
      '#default_value' => $settings['pinterestDropdownText'],
      '#size' => 30,
      '#description' => $this->t('Text displayed for the Pinterest share option.'),
      '#states' => [
        'visible' => [
          ':input[name="' . $plugin_id . '[settings_wrapper][pinterest]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // No specific validation needed.
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): array {
    $plugin_id = $this->getPluginId();
    $values = $form_state->getValue($plugin_id) ?? [];

    if (empty($values['enabled'])) {
      return ['enabled' => FALSE];
    }

    $settings_wrapper = $values['settings_wrapper'] ?? [];

    return [
      'enabled' => TRUE,
      'share' => (bool) ($settings_wrapper['share'] ?? TRUE),
      'facebook' => (bool) ($settings_wrapper['facebook'] ?? TRUE),
      'facebookDropdownText' => $settings_wrapper['facebookDropdownText'] ?? 'Facebook',
      'twitter' => (bool) ($settings_wrapper['twitter'] ?? TRUE),
      'twitterDropdownText' => $settings_wrapper['twitterDropdownText'] ?? 'Twitter',
      'pinterest' => (bool) ($settings_wrapper['pinterest'] ?? TRUE),
      'pinterestDropdownText' => $settings_wrapper['pinterestDropdownText'] ?? 'Pinterest',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildJsSettings(array $settings): array {
    if (!$this->isEnabled($settings)) {
      return [];
    }

    $merged = $this->mergeWithDefaults($settings);

    return [
      'share' => (bool) $merged['share'],
      'facebook' => (bool) $merged['facebook'],
      'facebookDropdownText' => $merged['facebookDropdownText'],
      'twitter' => (bool) $merged['twitter'],
      'twitterDropdownText' => $merged['twitterDropdownText'],
      'pinterest' => (bool) $merged['pinterest'],
      'pinterestDropdownText' => $merged['pinterestDropdownText'],
    ];
  }

}
