<?php

namespace Drupal\lightgallery_formatter\Plugin\Lightgallery;

use Drupal\Core\Form\FormStateInterface;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface;
use Drupal\lightgallery_formatter\LightgalleryPluginBase;

/**
 * Provides thumbnail navigation plugin for LightGallery.
 *
 * @LightgalleryPlugin(
 *   id = "thumbnail",
 *   label = @Translation("Thumbnail Navigation"),
 *   description = @Translation("Enable thumbnail navigation strip in the gallery."),
 *   weight = 0
 * )
 */
class Thumbnail extends LightgalleryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultSettings(): array {
    return [
      'enabled' => FALSE,
      'animateThumb' => TRUE,
      'image_style' => '',
      'thumbWidth' => 100,
      'thumbHeight' => '80px',
      'thumbMargin' => 5,
      'currentPagerPosition' => 'middle',
      'alignThumbnails' => 'middle',
      'toggleThumb' => FALSE,
      'appendThumbnailsTo' => '.lg-components',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(array $settings): bool {
    return !empty($settings['enabled']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array &$form, FormStateInterface $form_state, LightgalleryProfileInterface $profile): array {
    $settings = $this->mergeWithDefaults($profile->getPluginSettings('thumbnail'));
    $plugin_id = $this->getPluginId();

    $element = [
      '#type' => 'details',
      '#title' => $this->t('Thumbnail Navigation'),
      '#open' => !empty($settings['enabled']),
    ];

    $element['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable thumbnail plugin'),
      '#default_value' => $settings['enabled'],
      '#description' => $this->t('Check this to configure thumbnail navigation settings.'),
    ];

    $element['settings_wrapper'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="' . $plugin_id . '[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $element['settings_wrapper']['animateThumb'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Animate thumbnails'),
      '#default_value' => $settings['animateThumb'],
      '#description' => $this->t('Enable thumbnail animation.'),
    ];

    // Get available image styles.
    $image_styles = ['' => $this->t('- None -')];
    foreach ($this->entityTypeManager->getStorage('image_style')->loadMultiple() as $image_style) {
      $image_styles[$image_style->id()] = $image_style->label();
    }

    $element['settings_wrapper']['image_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Lightbox thumbnails (strip at bottom)'),
      '#description' => $this->t('Image style for the thumbnail strip at the bottom of the lightbox. This is separate from "Gallery thumbnails" configured in the field formatter display settings.'),
      '#options' => $image_styles,
      '#default_value' => $settings['image_style'],
    ];

    $element['settings_wrapper']['thumbWidth'] = [
      '#type' => 'number',
      '#title' => $this->t('Thumbnail width'),
      '#default_value' => $settings['thumbWidth'],
      '#min' => 20,
      '#max' => 500,
      '#description' => $this->t('Width of each thumbnail in pixels.'),
      '#field_suffix' => 'px',
    ];

    $element['settings_wrapper']['thumbHeight'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Thumbnail height'),
      '#default_value' => $settings['thumbHeight'],
      '#size' => 10,
      '#description' => $this->t('Height of each thumbnail (e.g., "80px", "100%").'),
    ];

    $element['settings_wrapper']['thumbMargin'] = [
      '#type' => 'number',
      '#title' => $this->t('Thumbnail margin'),
      '#default_value' => $settings['thumbMargin'],
      '#min' => 0,
      '#description' => $this->t('Spacing between thumbnails in pixels.'),
      '#field_suffix' => 'px',
    ];

    $element['settings_wrapper']['currentPagerPosition'] = [
      '#type' => 'select',
      '#title' => $this->t('Current thumbnail position'),
      '#default_value' => $settings['currentPagerPosition'],
      '#options' => [
        'left' => $this->t('Left'),
        'middle' => $this->t('Middle'),
        'right' => $this->t('Right'),
      ],
      '#description' => $this->t('Position of the currently selected thumbnail.'),
    ];

    $element['settings_wrapper']['alignThumbnails'] = [
      '#type' => 'select',
      '#title' => $this->t('Thumbnail alignment'),
      '#default_value' => $settings['alignThumbnails'],
      '#options' => [
        'left' => $this->t('Left'),
        'middle' => $this->t('Middle'),
        'right' => $this->t('Right'),
      ],
      '#description' => $this->t('Alignment when thumbnails width is less than gallery width.'),
    ];

    $element['settings_wrapper']['toggleThumb'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable toggle button'),
      '#default_value' => $settings['toggleThumb'],
      '#description' => $this->t('Show a button to toggle thumbnail visibility.'),
    ];

    $element['settings_wrapper']['appendThumbnailsTo'] = [
      '#type' => 'select',
      '#title' => $this->t('Append thumbnails to'),
      '#default_value' => $settings['appendThumbnailsTo'],
      '#options' => [
        '.lg-components' => $this->t('Gallery components (with transitions)'),
        '.lg-outer' => $this->t('Gallery outer (static, no transitions)'),
      ],
      '#description' => $this->t('Where thumbnails should be appended in the DOM.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $plugin_id = $this->getPluginId();

    $thumbWidth = $form_state->getValue([$plugin_id, 'settings_wrapper', 'thumbWidth']);
    if ($thumbWidth !== NULL && (!is_numeric($thumbWidth) || $thumbWidth < 20)) {
      $form_state->setErrorByName(
        "{$plugin_id}][settings_wrapper][thumbWidth",
        $this->t('Thumbnail width must be at least 20 pixels.')
      );
    }

    $thumbMargin = $form_state->getValue([$plugin_id, 'settings_wrapper', 'thumbMargin']);
    if ($thumbMargin !== NULL && (!is_numeric($thumbMargin) || $thumbMargin < 0)) {
      $form_state->setErrorByName(
        "{$plugin_id}][settings_wrapper][thumbMargin",
        $this->t('Thumbnail margin must be a positive number.')
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): array {
    $plugin_id = $this->getPluginId();
    $values = $form_state->getValue($plugin_id) ?? [];

    // If not enabled, return empty settings (or just enabled = false).
    if (empty($values['enabled'])) {
      return ['enabled' => FALSE];
    }

    $settings_wrapper = $values['settings_wrapper'] ?? [];

    return [
      'enabled' => TRUE,
      'animateThumb' => (bool) ($settings_wrapper['animateThumb'] ?? TRUE),
      'image_style' => $settings_wrapper['image_style'] ?? '',
      'thumbWidth' => (int) ($settings_wrapper['thumbWidth'] ?? 100),
      'thumbHeight' => $settings_wrapper['thumbHeight'] ?? '80px',
      'thumbMargin' => (int) ($settings_wrapper['thumbMargin'] ?? 5),
      'currentPagerPosition' => $settings_wrapper['currentPagerPosition'] ?? 'middle',
      'alignThumbnails' => $settings_wrapper['alignThumbnails'] ?? 'middle',
      'toggleThumb' => (bool) ($settings_wrapper['toggleThumb'] ?? FALSE),
      'appendThumbnailsTo' => $settings_wrapper['appendThumbnailsTo'] ?? '.lg-components',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildJsSettings(array $settings): array {
    // Only return settings if enabled.
    if (!$this->isEnabled($settings)) {
      return [];
    }

    $merged = $this->mergeWithDefaults($settings);

    return [
      'thumbnail' => TRUE,
      'animateThumb' => (bool) $merged['animateThumb'],
      'thumbWidth' => (int) $merged['thumbWidth'],
      'thumbHeight' => $merged['thumbHeight'],
      'thumbMargin' => (int) $merged['thumbMargin'],
      'currentPagerPosition' => $merged['currentPagerPosition'],
      'alignThumbnails' => $merged['alignThumbnails'],
      'toggleThumb' => (bool) $merged['toggleThumb'],
      'appendThumbnailsTo' => $merged['appendThumbnailsTo'],
    ];
  }

}
