<?php

namespace Drupal\lightgallery_formatter_test\Plugin\Lightgallery;

use Drupal\Core\Form\FormStateInterface;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface;
use Drupal\lightgallery_formatter\LightgalleryPluginBase;

/**
 * Test plugin for LightGallery third-party plugin testing.
 *
 * @LightgalleryPlugin(
 *   id = "test_plugin",
 *   label = @Translation("Test Plugin"),
 *   description = @Translation("A test plugin for third-party plugin testing."),
 *   weight = 100
 * )
 */
class TestPlugin extends LightgalleryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultSettings(): array {
    return [
      'enabled' => FALSE,
      'test_option' => 'default_value',
      'test_number' => 42,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array &$form, FormStateInterface $form_state, LightgalleryProfileInterface $profile): array {
    $settings = $this->mergeWithDefaults($profile->getPluginSettings($this->getPluginId()));

    $form['test_plugin_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Test Plugin Settings'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $form['test_plugin_settings']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable test plugin'),
      '#default_value' => $settings['enabled'],
      '#description' => $this->t('Check this to enable the test plugin.'),
    ];

    $form['test_plugin_settings']['test_option'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test option'),
      '#default_value' => $settings['test_option'],
      '#description' => $this->t('A test string option.'),
      '#states' => [
        'visible' => [
          ':input[name="test_plugin_settings[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['test_plugin_settings']['test_number'] = [
      '#type' => 'number',
      '#title' => $this->t('Test number'),
      '#default_value' => $settings['test_number'],
      '#min' => 0,
      '#description' => $this->t('A test numeric option.'),
      '#states' => [
        'visible' => [
          ':input[name="test_plugin_settings[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $form['test_plugin_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $settings = $form_state->getValue('test_plugin_settings');

    if ($settings['enabled']) {
      if (isset($settings['test_number']) && (!is_numeric($settings['test_number']) || $settings['test_number'] < 0)) {
        $form_state->setErrorByName('test_plugin_settings][test_number', $this->t('Test number must be a positive number.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): array {
    return $form_state->getValue('test_plugin_settings') ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function buildJsSettings(array $settings): array {
    if (!$this->isEnabled($settings)) {
      return [];
    }
    return [
      'testPlugin' => TRUE,
      'testOption' => $settings['test_option'] ?? $this->defaultSettings()['test_option'],
      'testNumber' => (int) ($settings['test_number'] ?? $this->defaultSettings()['test_number']),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(array $settings): bool {
    return (bool) ($settings['enabled'] ?? FALSE);
  }

}
