<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;

/**
 * Tests module installation and default configuration.
 *
 * @group lightgallery_formatter
 */
class LightgalleryFormatterInstallTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
  }

  /**
   * Test that the module installs correctly.
   */
  public function testModuleInstallation(): void {
    // Check that the module is enabled.
    $module_handler = $this->container->get('module_handler');
    $this->assertTrue($module_handler->moduleExists('lightgallery_formatter'), 'Module is installed.');
  }

  /**
   * Test that default profile is created on install.
   */
  public function testDefaultProfileExists(): void {
    $storage = $this->container->get('entity_type.manager')->getStorage('lightgallery_profile');
    $profile = $storage->load('default');

    $this->assertNotNull($profile, 'Default profile exists after installation.');
    $this->assertInstanceOf(LightgalleryProfile::class, $profile, 'Profile is correct entity type.');
    $this->assertEquals('Default', $profile->label(), 'Default profile has correct label.');
    $this->assertTrue($profile->getStatus(), 'Default profile is enabled.');
  }

  /**
   * Test default profile has correct general settings.
   */
  public function testDefaultProfileGeneralSettings(): void {
    $storage = $this->container->get('entity_type.manager')->getStorage('lightgallery_profile');
    $profile = $storage->load('default');

    // Get general settings from the profile.
    $general_settings = $profile->getPluginSettings('general');

    // Use plugin manager to get defaults and merge.
    $plugin_manager = $this->container->get('plugin.manager.lightgallery');
    $general_plugin = $plugin_manager->createInstance('general', $general_settings);
    $merged_settings = array_merge($general_plugin->defaultSettings(), $general_settings);

    // Check some key default values.
    $this->assertTrue($merged_settings['loop'], 'Loop is enabled by default.');
    $this->assertEquals(400, $merged_settings['speed'], 'Default speed is 400ms.');
    $this->assertEquals('lg-slide', $merged_settings['mode'], 'Default mode is lg-slide.');
    $this->assertTrue($merged_settings['escKey'], 'ESC key is enabled by default.');
    $this->assertTrue($merged_settings['counter'], 'Counter is enabled by default.');
  }

  /**
   * Test LightgalleryProfile entity type is registered.
   */
  public function testEntityTypeRegistered(): void {
    $entity_type_manager = $this->container->get('entity_type.manager');

    $this->assertTrue($entity_type_manager->hasDefinition('lightgallery_profile'), 'Entity type is registered.');

    $definition = $entity_type_manager->getDefinition('lightgallery_profile');
    $this->assertEquals('lightgallery_formatter', $definition->getProvider(), 'Entity type provider is correct.');
    $this->assertEquals('configuration', $definition->getGroup(), 'Entity type is a config entity.');
  }

  /**
   * Test field formatter plugin is registered.
   */
  public function testFieldFormatterRegistered(): void {
    $formatter_manager = $this->container->get('plugin.manager.field.formatter');

    $this->assertTrue($formatter_manager->hasDefinition('lightgallery_formatter'), 'Field formatter is registered.');

    $definition = $formatter_manager->getDefinition('lightgallery_formatter');
    $this->assertEquals('LightGallery', $definition['label']->render(), 'Formatter has correct label.');
  }

  /**
   * Test LightGallery plugin manager is registered.
   */
  public function testPluginManagerRegistered(): void {
    $plugin_manager = $this->container->get('plugin.manager.lightgallery');
    $this->assertNotNull($plugin_manager, 'Plugin manager is registered.');

    $definitions = $plugin_manager->getDefinitions();
    $this->assertArrayHasKey('general', $definitions, 'General plugin is registered.');
    $this->assertArrayHasKey('thumbnail', $definitions, 'Thumbnail plugin is registered.');
  }

}
