<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\KernelTests\KernelTestBase;

/**
 * Tests default settings for LightGallery plugins.
 *
 * @group lightgallery_formatter
 */
class LightgalleryPluginDefaultsTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * The plugin manager.
   *
   * @var \Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryPluginManager
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
    $this->pluginManager = $this->container->get('plugin.manager.lightgallery');
  }

  /**
   * Test General plugin defaultSettings returns all expected keys.
   */
  public function testGeneralPluginDefaultSettingsKeys(): void {
    $plugin = $this->pluginManager->createInstance('general');
    $defaults = $plugin->defaultSettings();

    $expected_keys = [
      'licenseKey',
      'loop',
      'preload',
      'speed',
      'mode',
      'closable',
      'closeOnTap',
      'escKey',
      'keyPress',
      'showCloseIcon',
      'showMaximizeIcon',
      'hideBarsDelay',
      'counter',
      'download',
      'enableDrag',
      'enableSwipe',
      'mousewheel',
      'allowMediaOverlap',
    ];

    foreach ($expected_keys as $key) {
      $this->assertArrayHasKey($key, $defaults, "General plugin has default for '{$key}'.");
    }

    // Ensure no extraneous keys.
    $this->assertCount(
      count($expected_keys),
      $defaults,
      'General plugin has exactly the expected number of default settings.'
    );
  }

  /**
   * Test General plugin default values match specification.
   */
  public function testGeneralPluginDefaultValues(): void {
    $plugin = $this->pluginManager->createInstance('general');
    $defaults = $plugin->defaultSettings();

    // Test specific default values.
    $this->assertTrue($defaults['loop'], 'Default loop is TRUE.');
    $this->assertEquals(2, $defaults['preload'], 'Default preload is 2.');
    $this->assertEquals(400, $defaults['speed'], 'Default speed is 400.');
    $this->assertEquals('lg-slide', $defaults['mode'], 'Default mode is lg-slide.');
    $this->assertTrue($defaults['closable'], 'Default closable is TRUE.');
    $this->assertTrue($defaults['closeOnTap'], 'Default closeOnTap is TRUE.');
    $this->assertTrue($defaults['escKey'], 'Default escKey is TRUE.');
    $this->assertTrue($defaults['keyPress'], 'Default keyPress is TRUE.');
    $this->assertTrue($defaults['showCloseIcon'], 'Default showCloseIcon is TRUE.');
    $this->assertFalse($defaults['showMaximizeIcon'], 'Default showMaximizeIcon is FALSE.');
    $this->assertEquals(0, $defaults['hideBarsDelay'], 'Default hideBarsDelay is 0.');
    $this->assertTrue($defaults['counter'], 'Default counter is TRUE.');
    $this->assertTrue($defaults['download'], 'Default download is TRUE.');
    $this->assertTrue($defaults['enableDrag'], 'Default enableDrag is TRUE.');
    $this->assertTrue($defaults['enableSwipe'], 'Default enableSwipe is TRUE.');
    $this->assertFalse($defaults['mousewheel'], 'Default mousewheel is FALSE.');
    $this->assertTrue($defaults['allowMediaOverlap'], 'Default allowMediaOverlap is TRUE.');
  }

  /**
   * Test Thumbnail plugin defaultSettings returns all expected keys.
   */
  public function testThumbnailPluginDefaultSettingsKeys(): void {
    $plugin = $this->pluginManager->createInstance('thumbnail');
    $defaults = $plugin->defaultSettings();

    $expected_keys = [
      'enabled',
      'animateThumb',
      'image_style',
      'thumbWidth',
      'thumbHeight',
      'thumbMargin',
      'currentPagerPosition',
      'alignThumbnails',
      'toggleThumb',
      'appendThumbnailsTo',
    ];

    $this->assertCount(10, $expected_keys, 'Expected 10 thumbnail settings keys.');

    foreach ($expected_keys as $key) {
      $this->assertArrayHasKey($key, $defaults, "Thumbnail plugin has default for '{$key}'.");
    }

    // Ensure no extraneous keys.
    $this->assertCount(
      count($expected_keys),
      $defaults,
      'Thumbnail plugin has exactly the expected number of default settings.'
    );
  }

  /**
   * Test Thumbnail plugin default values match specification.
   */
  public function testThumbnailPluginDefaultValues(): void {
    $plugin = $this->pluginManager->createInstance('thumbnail');
    $defaults = $plugin->defaultSettings();

    $this->assertFalse($defaults['enabled'], 'Default enabled is FALSE.');
    $this->assertTrue($defaults['animateThumb'], 'Default animateThumb is TRUE.');
    $this->assertEquals('', $defaults['image_style'], 'Default image_style is empty string.');
    $this->assertEquals(100, $defaults['thumbWidth'], 'Default thumbWidth is 100.');
    $this->assertEquals('80px', $defaults['thumbHeight'], 'Default thumbHeight is 80px.');
    $this->assertEquals(5, $defaults['thumbMargin'], 'Default thumbMargin is 5.');
    $this->assertEquals('middle', $defaults['currentPagerPosition'], 'Default currentPagerPosition is middle.');
    $this->assertEquals('middle', $defaults['alignThumbnails'], 'Default alignThumbnails is middle.');
    $this->assertFalse($defaults['toggleThumb'], 'Default toggleThumb is FALSE.');
    $this->assertEquals('.lg-components', $defaults['appendThumbnailsTo'], 'Default appendThumbnailsTo is .lg-components.');
  }

  /**
   * Test defaultConfiguration returns same as defaultSettings.
   */
  public function testDefaultConfigurationMatchesDefaultSettings(): void {
    $general = $this->pluginManager->createInstance('general');
    $thumbnail = $this->pluginManager->createInstance('thumbnail');

    $this->assertEquals(
      $general->defaultSettings(),
      $general->defaultConfiguration(),
      'General defaultConfiguration matches defaultSettings.'
    );

    $this->assertEquals(
      $thumbnail->defaultSettings(),
      $thumbnail->defaultConfiguration(),
      'Thumbnail defaultConfiguration matches defaultSettings.'
    );
  }

  /**
   * Test that default values are of correct types.
   */
  public function testGeneralPluginDefaultTypes(): void {
    $plugin = $this->pluginManager->createInstance('general');
    $defaults = $plugin->defaultSettings();

    // Boolean settings.
    $boolean_keys = [
      'loop',
      'closable',
      'closeOnTap',
      'escKey',
      'keyPress',
      'showCloseIcon',
      'showMaximizeIcon',
      'counter',
      'download',
      'enableDrag',
      'enableSwipe',
      'mousewheel',
      'allowMediaOverlap',
    ];

    foreach ($boolean_keys as $key) {
      $this->assertIsBool($defaults[$key], "'{$key}' default is boolean.");
    }

    // Integer settings.
    $int_keys = ['preload', 'speed', 'hideBarsDelay'];
    foreach ($int_keys as $key) {
      $this->assertIsInt($defaults[$key], "'{$key}' default is integer.");
    }

    // String settings.
    $this->assertIsString($defaults['mode'], "'mode' default is string.");
  }

  /**
   * Test that thumbnail default values are of correct types.
   */
  public function testThumbnailPluginDefaultTypes(): void {
    $plugin = $this->pluginManager->createInstance('thumbnail');
    $defaults = $plugin->defaultSettings();

    // Boolean settings.
    $this->assertIsBool($defaults['enabled'], "'enabled' default is boolean.");
    $this->assertIsBool($defaults['animateThumb'], "'animateThumb' default is boolean.");
    $this->assertIsBool($defaults['toggleThumb'], "'toggleThumb' default is boolean.");

    // Integer settings.
    $this->assertIsInt($defaults['thumbWidth'], "'thumbWidth' default is integer.");
    $this->assertIsInt($defaults['thumbMargin'], "'thumbMargin' default is integer.");

    // String settings.
    $this->assertIsString($defaults['image_style'], "'image_style' default is string.");
    $this->assertIsString($defaults['thumbHeight'], "'thumbHeight' default is string.");
    $this->assertIsString($defaults['currentPagerPosition'], "'currentPagerPosition' default is string.");
    $this->assertIsString($defaults['alignThumbnails'], "'alignThumbnails' default is string.");
    $this->assertIsString($defaults['appendThumbnailsTo'], "'appendThumbnailsTo' default is string.");
  }

}
