<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\lightgallery_formatter\LightgalleryPluginInterface;

/**
 * Tests plugin discovery for LightGallery plugins.
 *
 * @group lightgallery_formatter
 */
class LightgalleryPluginDiscoveryTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * The plugin manager.
   *
   * @var \Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryPluginManager
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
    $this->pluginManager = $this->container->get('plugin.manager.lightgallery');
  }

  /**
   * Test that plugin manager finds built-in plugins.
   */
  public function testBuiltInPluginsDiscovered(): void {
    $definitions = $this->pluginManager->getDefinitions();

    $this->assertArrayHasKey('general', $definitions, 'General plugin is discovered.');
    $this->assertArrayHasKey('thumbnail', $definitions, 'Thumbnail plugin is discovered.');
    $this->assertArrayHasKey('zoom', $definitions, 'Zoom plugin is discovered.');
    $this->assertArrayHasKey('fullscreen', $definitions, 'Fullscreen plugin is discovered.');
    $this->assertArrayHasKey('pager', $definitions, 'Pager plugin is discovered.');
    $this->assertArrayHasKey('relative_caption', $definitions, 'Relative Caption plugin is discovered.');
    $this->assertArrayHasKey('rotate', $definitions, 'Rotate plugin is discovered.');
    $this->assertArrayHasKey('autoplay', $definitions, 'Autoplay plugin is discovered.');
    $this->assertArrayHasKey('medium_zoom', $definitions, 'Medium Zoom plugin is discovered.');
    $this->assertArrayHasKey('share', $definitions, 'Share plugin is discovered.');
    $this->assertArrayHasKey('hash', $definitions, 'Hash plugin is discovered.');
    $this->assertCount(11, $definitions, 'Exactly 11 built-in plugins are discovered.');
  }

  /**
   * Test that plugin definitions have correct metadata.
   */
  public function testPluginDefinitionMetadata(): void {
    $definitions = $this->pluginManager->getDefinitions();

    // Test General plugin metadata.
    $general = $definitions['general'];
    $this->assertEquals('general', $general['id'], 'General plugin ID is correct.');
    $this->assertNotEmpty($general['label'], 'General plugin has a label.');
    $this->assertNotEmpty($general['description'], 'General plugin has a description.');
    $this->assertArrayHasKey('weight', $general, 'General plugin has weight.');
    $this->assertEquals(-100, $general['weight'], 'General plugin has correct weight.');

    // Test Thumbnail plugin metadata.
    $thumbnail = $definitions['thumbnail'];
    $this->assertEquals('thumbnail', $thumbnail['id'], 'Thumbnail plugin ID is correct.');
    $this->assertNotEmpty($thumbnail['label'], 'Thumbnail plugin has a label.');
    $this->assertNotEmpty($thumbnail['description'], 'Thumbnail plugin has a description.');
    $this->assertArrayHasKey('weight', $thumbnail, 'Thumbnail plugin has weight.');
    $this->assertEquals(0, $thumbnail['weight'], 'Thumbnail plugin has correct weight.');

    // Test Zoom plugin metadata.
    $zoom = $definitions['zoom'];
    $this->assertEquals('zoom', $zoom['id'], 'Zoom plugin ID is correct.');
    $this->assertNotEmpty($zoom['label'], 'Zoom plugin has a label.');
    $this->assertNotEmpty($zoom['description'], 'Zoom plugin has a description.');
    $this->assertArrayHasKey('weight', $zoom, 'Zoom plugin has weight.');
    $this->assertEquals(10, $zoom['weight'], 'Zoom plugin has correct weight.');

    // Test Fullscreen plugin metadata.
    $fullscreen = $definitions['fullscreen'];
    $this->assertEquals('fullscreen', $fullscreen['id'], 'Fullscreen plugin ID is correct.');
    $this->assertNotEmpty($fullscreen['label'], 'Fullscreen plugin has a label.');
    $this->assertNotEmpty($fullscreen['description'], 'Fullscreen plugin has a description.');
    $this->assertArrayHasKey('weight', $fullscreen, 'Fullscreen plugin has weight.');
    $this->assertEquals(20, $fullscreen['weight'], 'Fullscreen plugin has correct weight.');

    // Test Pager plugin metadata.
    $pager = $definitions['pager'];
    $this->assertEquals('pager', $pager['id'], 'Pager plugin ID is correct.');
    $this->assertNotEmpty($pager['label'], 'Pager plugin has a label.');
    $this->assertNotEmpty($pager['description'], 'Pager plugin has a description.');
    $this->assertArrayHasKey('weight', $pager, 'Pager plugin has weight.');
    $this->assertEquals(25, $pager['weight'], 'Pager plugin has correct weight.');

    // Test Relative Caption plugin metadata.
    $relative_caption = $definitions['relative_caption'];
    $this->assertEquals('relative_caption', $relative_caption['id'], 'Relative Caption plugin ID is correct.');
    $this->assertNotEmpty($relative_caption['label'], 'Relative Caption plugin has a label.');
    $this->assertNotEmpty($relative_caption['description'], 'Relative Caption plugin has a description.');
    $this->assertArrayHasKey('weight', $relative_caption, 'Relative Caption plugin has weight.');
    $this->assertEquals(30, $relative_caption['weight'], 'Relative Caption plugin has correct weight.');

    // Test Rotate plugin metadata.
    $rotate = $definitions['rotate'];
    $this->assertEquals('rotate', $rotate['id'], 'Rotate plugin ID is correct.');
    $this->assertNotEmpty($rotate['label'], 'Rotate plugin has a label.');
    $this->assertNotEmpty($rotate['description'], 'Rotate plugin has a description.');
    $this->assertArrayHasKey('weight', $rotate, 'Rotate plugin has weight.');
    $this->assertEquals(35, $rotate['weight'], 'Rotate plugin has correct weight.');

    // Test Autoplay plugin metadata.
    $autoplay = $definitions['autoplay'];
    $this->assertEquals('autoplay', $autoplay['id'], 'Autoplay plugin ID is correct.');
    $this->assertNotEmpty($autoplay['label'], 'Autoplay plugin has a label.');
    $this->assertNotEmpty($autoplay['description'], 'Autoplay plugin has a description.');
    $this->assertArrayHasKey('weight', $autoplay, 'Autoplay plugin has weight.');
    $this->assertEquals(40, $autoplay['weight'], 'Autoplay plugin has correct weight.');

    // Test Medium Zoom plugin metadata.
    $medium_zoom = $definitions['medium_zoom'];
    $this->assertEquals('medium_zoom', $medium_zoom['id'], 'Medium Zoom plugin ID is correct.');
    $this->assertNotEmpty($medium_zoom['label'], 'Medium Zoom plugin has a label.');
    $this->assertNotEmpty($medium_zoom['description'], 'Medium Zoom plugin has a description.');
    $this->assertArrayHasKey('weight', $medium_zoom, 'Medium Zoom plugin has weight.');
    $this->assertEquals(45, $medium_zoom['weight'], 'Medium Zoom plugin has correct weight.');

    // Test Share plugin metadata.
    $share = $definitions['share'];
    $this->assertEquals('share', $share['id'], 'Share plugin ID is correct.');
    $this->assertNotEmpty($share['label'], 'Share plugin has a label.');
    $this->assertNotEmpty($share['description'], 'Share plugin has a description.');
    $this->assertArrayHasKey('weight', $share, 'Share plugin has weight.');
    $this->assertEquals(50, $share['weight'], 'Share plugin has correct weight.');

    // Test Hash plugin metadata.
    $hash = $definitions['hash'];
    $this->assertEquals('hash', $hash['id'], 'Hash plugin ID is correct.');
    $this->assertNotEmpty($hash['label'], 'Hash plugin has a label.');
    $this->assertNotEmpty($hash['description'], 'Hash plugin has a description.');
    $this->assertArrayHasKey('weight', $hash, 'Hash plugin has weight.');
    $this->assertEquals(55, $hash['weight'], 'Hash plugin has correct weight.');
  }

  /**
   * Test that plugins are sorted by weight.
   */
  public function testPluginsSortedByWeight(): void {
    $sorted = $this->pluginManager->getSortedDefinitions();
    $plugin_ids = array_keys($sorted);

    // Plugin weights in order:
    // General: -100, Thumbnail: 0, Zoom: 10, Fullscreen: 20, Pager: 25,
    // RelativeCaption: 30, Rotate: 35, Autoplay: 40, MediumZoom: 45,
    // Share: 50, Hash: 55.
    $this->assertEquals('general', $plugin_ids[0], 'General plugin comes first (weight -100).');
    $this->assertEquals('thumbnail', $plugin_ids[1], 'Thumbnail plugin comes second (weight 0).');
    $this->assertEquals('zoom', $plugin_ids[2], 'Zoom plugin comes third (weight 10).');
    $this->assertEquals('fullscreen', $plugin_ids[3], 'Fullscreen plugin comes fourth (weight 20).');
    $this->assertEquals('pager', $plugin_ids[4], 'Pager plugin comes fifth (weight 25).');
    $this->assertEquals('relative_caption', $plugin_ids[5], 'Relative Caption plugin comes sixth (weight 30).');
    $this->assertEquals('rotate', $plugin_ids[6], 'Rotate plugin comes seventh (weight 35).');
    $this->assertEquals('autoplay', $plugin_ids[7], 'Autoplay plugin comes eighth (weight 40).');
    $this->assertEquals('medium_zoom', $plugin_ids[8], 'Medium Zoom plugin comes ninth (weight 45).');
    $this->assertEquals('share', $plugin_ids[9], 'Share plugin comes tenth (weight 50).');
    $this->assertEquals('hash', $plugin_ids[10], 'Hash plugin comes eleventh (weight 55).');
  }

  /**
   * Test createInstances method.
   */
  public function testCreateInstances(): void {
    $instances = $this->pluginManager->createInstances();

    $this->assertArrayHasKey('general', $instances, 'General plugin instance created.');
    $this->assertArrayHasKey('thumbnail', $instances, 'Thumbnail plugin instance created.');
    $this->assertArrayHasKey('zoom', $instances, 'Zoom plugin instance created.');
    $this->assertArrayHasKey('fullscreen', $instances, 'Fullscreen plugin instance created.');
    $this->assertArrayHasKey('pager', $instances, 'Pager plugin instance created.');
    $this->assertArrayHasKey('relative_caption', $instances, 'Relative Caption plugin instance created.');
    $this->assertArrayHasKey('rotate', $instances, 'Rotate plugin instance created.');
    $this->assertArrayHasKey('autoplay', $instances, 'Autoplay plugin instance created.');
    $this->assertArrayHasKey('medium_zoom', $instances, 'Medium Zoom plugin instance created.');
    $this->assertArrayHasKey('share', $instances, 'Share plugin instance created.');
    $this->assertArrayHasKey('hash', $instances, 'Hash plugin instance created.');

    foreach ($instances as $plugin_id => $instance) {
      $this->assertInstanceOf(
        LightgalleryPluginInterface::class,
        $instance,
        "Plugin '{$plugin_id}' implements LightgalleryPluginInterface."
      );
    }
  }

  /**
   * Test createInstances with configuration.
   */
  public function testCreateInstancesWithConfiguration(): void {
    $plugin_settings = [
      'general' => ['loop' => FALSE, 'speed' => 800],
      'thumbnail' => ['enabled' => TRUE, 'thumbMargin' => 10],
      'zoom' => ['enabled' => TRUE, 'scale' => 2],
      'fullscreen' => ['enabled' => TRUE],
      'pager' => ['enabled' => TRUE, 'pager' => TRUE],
      'relative_caption' => ['enabled' => TRUE, 'relativeCaption' => TRUE],
      'rotate' => ['enabled' => TRUE, 'rotateSpeed' => 500],
      'autoplay' => ['enabled' => TRUE, 'slideShowInterval' => 3000],
      'medium_zoom' => ['enabled' => TRUE, 'margin' => 50],
      'share' => ['enabled' => TRUE, 'facebook' => TRUE],
      'hash' => ['enabled' => TRUE, 'galleryId' => 'test-gallery'],
    ];

    $instances = $this->pluginManager->createInstances($plugin_settings);

    $general_config = $instances['general']->getConfiguration();
    $this->assertFalse($general_config['loop'], 'General plugin received configuration.');
    $this->assertEquals(800, $general_config['speed'], 'General plugin speed configured.');

    $thumbnail_config = $instances['thumbnail']->getConfiguration();
    $this->assertTrue($thumbnail_config['enabled'], 'Thumbnail plugin received configuration.');
    $this->assertEquals(10, $thumbnail_config['thumbMargin'], 'Thumbnail plugin margin configured.');

    $zoom_config = $instances['zoom']->getConfiguration();
    $this->assertTrue($zoom_config['enabled'], 'Zoom plugin received configuration.');
    $this->assertEquals(2, $zoom_config['scale'], 'Zoom plugin scale configured.');

    $fullscreen_config = $instances['fullscreen']->getConfiguration();
    $this->assertTrue($fullscreen_config['enabled'], 'Fullscreen plugin received configuration.');

    $pager_config = $instances['pager']->getConfiguration();
    $this->assertTrue($pager_config['enabled'], 'Pager plugin received configuration.');
    $this->assertTrue($pager_config['pager'], 'Pager plugin pager option configured.');

    $relative_caption_config = $instances['relative_caption']->getConfiguration();
    $this->assertTrue($relative_caption_config['enabled'], 'Relative Caption plugin received configuration.');
    $this->assertTrue($relative_caption_config['relativeCaption'], 'Relative Caption option configured.');

    $rotate_config = $instances['rotate']->getConfiguration();
    $this->assertTrue($rotate_config['enabled'], 'Rotate plugin received configuration.');
    $this->assertEquals(500, $rotate_config['rotateSpeed'], 'Rotate plugin speed configured.');

    $autoplay_config = $instances['autoplay']->getConfiguration();
    $this->assertTrue($autoplay_config['enabled'], 'Autoplay plugin received configuration.');
    $this->assertEquals(3000, $autoplay_config['slideShowInterval'], 'Autoplay interval configured.');

    $medium_zoom_config = $instances['medium_zoom']->getConfiguration();
    $this->assertTrue($medium_zoom_config['enabled'], 'Medium Zoom plugin received configuration.');
    $this->assertEquals(50, $medium_zoom_config['margin'], 'Medium Zoom margin configured.');

    $share_config = $instances['share']->getConfiguration();
    $this->assertTrue($share_config['enabled'], 'Share plugin received configuration.');
    $this->assertTrue($share_config['facebook'], 'Share plugin Facebook configured.');

    $hash_config = $instances['hash']->getConfiguration();
    $this->assertTrue($hash_config['enabled'], 'Hash plugin received configuration.');
    $this->assertEquals('test-gallery', $hash_config['galleryId'], 'Hash plugin galleryId configured.');
  }

  /**
   * Test plugin provider is correct.
   */
  public function testPluginProvider(): void {
    $definitions = $this->pluginManager->getDefinitions();

    foreach ($definitions as $plugin_id => $definition) {
      $this->assertEquals(
        'lightgallery_formatter',
        $definition['provider'],
        "Plugin '{$plugin_id}' has correct provider."
      );
    }
  }

}
