<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\user\Entity\Role;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\user\Entity\User;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;

/**
 * Tests plugin settings for LightgalleryProfile configuration entities.
 *
 * @group lightgallery_formatter
 */
class LightgalleryProfilePluginSettingsTest extends EntityKernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter', 'user', 'image'];

  /**
   * The entity storage handler.
   *
   * @var \Drupal\Core\Config\Entity\ConfigEntityStorageInterface
   */
  protected $storage;

  /**
   * A user with administrative permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * The LightGallery plugin manager.
   *
   * @var \Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryPluginManager
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter', 'image']);

    // Create a regular user.
    $user = User::create([
      'name' => 'authorized_user',
      'mail' => 'authorized@example.com',
      'status' => 1,
    ]);
    $user->save();

    // Create a custom role with only the required permission.
    $role = Role::create([
      'id' => 'lightgallery_admin',
      'label' => 'Lightgallery Administrator',
    ]);
    $role->grantPermission('administer lightgallery_formatter');
    $role->save();

    // Assign the custom role to the user.
    $user->addRole('lightgallery_admin');
    $user->save();
    $this->adminUser = $user;

    // Set up storage.
    $this->storage = $this->container->get('entity_type.manager')->getStorage('lightgallery_profile');

    // Get plugin manager.
    $this->pluginManager = $this->container->get('plugin.manager.lightgallery');
  }

  /**
   * Test plugin settings functionality.
   */
  public function testPluginSettings() {
    // Create a new profile with plugin settings.
    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
      'status' => TRUE,
    ]);

    // Set thumbnail plugin settings.
    $thumbnail_settings = [
      'enabled' => TRUE,
      'animateThumb' => FALSE,
      'image_style' => 'thumbnail',
      'thumbWidth' => 120,
      'thumbHeight' => '100px',
      'thumbMargin' => 10,
      'currentPagerPosition' => 'left',
      'alignThumbnails' => 'right',
      'toggleThumb' => TRUE,
      'appendThumbnailsTo' => '.lg-outer',
    ];

    $profile->setPluginSettings('thumbnail', $thumbnail_settings);

    // Set general plugin settings with all available options.
    $general_settings = [
      'loop' => FALSE,
      'preload' => 5,
      'speed' => 600,
      'mode' => 'lg-fade',
      'closable' => FALSE,
      'closeOnTap' => FALSE,
      'escKey' => FALSE,
      'keyPress' => FALSE,
      'showCloseIcon' => FALSE,
      'showMaximizeIcon' => TRUE,
      'hideBarsDelay' => 3000,
      'counter' => FALSE,
      'download' => FALSE,
      'enableDrag' => FALSE,
      'enableSwipe' => FALSE,
      'mousewheel' => TRUE,
    ];

    $profile->setPluginSettings('general', $general_settings);
    $profile->save();

    // Reload from storage.
    $this->storage->resetCache();
    $reloaded = $this->storage->load('test_profile');

    // Assert basic values.
    $this->assertNotNull($reloaded, 'Profile was saved and can be loaded.');
    $this->assertEquals('Test Profile', $reloaded->label(), 'Label is correct.');
    $this->assertTrue($reloaded->getStatus(), 'Status is enabled.');

    // Check thumbnail plugin settings.
    $loaded_thumbnail_settings = $reloaded->getPluginSettings('thumbnail');
    $this->assertEquals($thumbnail_settings, $loaded_thumbnail_settings, 'Thumbnail plugin settings are correctly saved and loaded.');

    // Check general plugin settings.
    $loaded_general_settings = $reloaded->getPluginSettings('general');
    $this->assertEquals($general_settings, $loaded_general_settings, 'General plugin settings are correctly saved and loaded.');

    // Verify each general setting value individually.
    foreach ($general_settings as $key => $value) {
      $this->assertEquals($value, $loaded_general_settings[$key], "General setting '{$key}' is correctly saved and loaded.");
    }

    // Test that plugin manager can create instances with these settings.
    $thumbnail_plugin = $this->pluginManager->createInstance('thumbnail', $loaded_thumbnail_settings);
    $this->assertTrue($thumbnail_plugin->isEnabled($loaded_thumbnail_settings), 'Thumbnail plugin is enabled.');

    // Test default settings from plugin.
    $general_plugin = $this->pluginManager->createInstance('general');
    $default_settings = $general_plugin->defaultSettings();
    $profile2 = LightgalleryProfile::create([
      'id' => 'test_profile2',
      'label' => 'Test Profile 2',
    ]);

    // Get settings and merge with defaults using plugin.
    $empty_settings = $profile2->getPluginSettings('general');
    $merged = array_merge($default_settings, $empty_settings);
    $this->assertEquals($default_settings, $merged, 'Default settings are correctly applied for empty profile.');

    // Test disabling plugin by setting empty array.
    $profile2->setPluginSettings('thumbnail', []);
    $disabled_settings = $profile2->getPluginSettings('thumbnail');
    $this->assertEmpty($disabled_settings, 'Plugin can be disabled by setting empty array.');
  }

  /**
   * Test plugin settings validation via plugin.
   */
  public function testPluginSettingsValidation() {
    $profile = LightgalleryProfile::create([
      'id' => 'validation_test',
      'label' => 'Validation Test',
    ]);

    // Test with valid settings.
    $valid_settings = [
      'enabled' => TRUE,
      'animateThumb' => TRUE,
      'image_style' => 'medium',
      'thumbMargin' => 10,
    ];

    $profile->setPluginSettings('thumbnail', $valid_settings);

    // Use plugin to get merged settings with defaults.
    $thumbnail_plugin = $this->pluginManager->createInstance('thumbnail', $valid_settings);
    $validated = array_merge($thumbnail_plugin->defaultSettings(), $valid_settings);

    // Should include defaults for missing values.
    $this->assertTrue(isset($validated['toggleThumb']), 'Default value for toggleThumb is set.');
    $this->assertTrue(isset($validated['appendThumbnailsTo']), 'Default value for appendThumbnailsTo is set.');

    // Test general settings validation with partial data.
    $general_valid_settings = [
      'loop' => TRUE,
      'preload' => 3,
      'speed' => 500,
      'mode' => 'lg-slide',
    ];

    $profile->setPluginSettings('general', $general_valid_settings);

    // Merge expected values with defaults for comparison.
    $general_plugin = $this->pluginManager->createInstance('general', $general_valid_settings);
    $default_general_settings = $general_plugin->defaultSettings();
    $expected_with_defaults = array_merge($default_general_settings, $general_valid_settings);

    $general_validated = array_merge($default_general_settings, $general_valid_settings);

    foreach ($expected_with_defaults as $key => $value) {
      $this->assertEquals($value, $general_validated[$key], "Validated general setting '{$key}' matches expected.");
    }
  }

  /**
   * Test that missing general settings use defaults.
   */
  public function testGeneralSettingsDefaults() {
    $profile = LightgalleryProfile::create([
      'id' => 'defaults_test',
      'label' => 'Defaults Test',
    ]);
    $profile->save();

    // Get defaults from plugin.
    $general_plugin = $this->pluginManager->createInstance('general');
    $defaults = $general_plugin->defaultSettings();

    // Get profile settings and merge with defaults.
    $profile_settings = $profile->getPluginSettings('general');
    $validated = array_merge($defaults, $profile_settings);

    foreach ($defaults as $key => $value) {
      $this->assertEquals($value, $validated[$key], "Default value for '{$key}' is applied.");
    }
  }

  /**
   * Test plugin JS settings building.
   */
  public function testPluginJsSettings() {
    $general_settings = [
      'loop' => FALSE,
      'preload' => 3,
      'speed' => 500,
      'mode' => 'lg-fade',
      'closable' => TRUE,
      'closeOnTap' => TRUE,
      'escKey' => TRUE,
      'keyPress' => TRUE,
      'showCloseIcon' => TRUE,
      'showMaximizeIcon' => FALSE,
      'hideBarsDelay' => 0,
      'counter' => TRUE,
      'download' => TRUE,
      'enableDrag' => TRUE,
      'enableSwipe' => TRUE,
      'mousewheel' => FALSE,
    ];

    $general_plugin = $this->pluginManager->createInstance('general', $general_settings);
    $js_settings = $general_plugin->buildJsSettings($general_settings);

    $this->assertFalse($js_settings['loop'], 'Loop JS setting is correct.');
    $this->assertEquals(3, $js_settings['preload'], 'Preload JS setting is correct.');
    $this->assertEquals(500, $js_settings['speed'], 'Speed JS setting is correct.');
    $this->assertEquals('lg-fade', $js_settings['mode'], 'Mode JS setting is correct.');
  }

}
