<?php

namespace Drupal\lightgallery_formatter\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ExtensionPathResolver;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\lightgallery_formatter\LightgalleryPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for the LightGallery profile add and edit forms.
 */
class LightgalleryProfileForm extends EntityForm {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The extension path resolver.
   *
   * @var \Drupal\Core\Extension\ExtensionPathResolver
   */
  protected $extensionPathResolver;

  /**
   * The module extension list.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * The app root path.
   *
   * @var string
   */
  protected $appRoot;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The preview builder service (if available).
   *
   * @var \Drupal\lightgallery_formatter_preview\PreviewBuilder|null
   */
  protected $previewBuilder;

  /**
   * The LightGallery plugin manager.
   *
   * @var \Drupal\lightgallery_formatter\LightgalleryPluginManager
   */
  protected LightgalleryPluginManager $pluginManager;

  /**
   * Constructs a new LightgalleryProfileForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ExtensionPathResolver $extension_path_resolver
   *   The extension path resolver.
   * @param \Drupal\Core\Extension\ModuleExtensionList $module_extension_list
   *   The module extension list.
   * @param string $app_root
   *   The app root path.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\lightgallery_formatter\LightgalleryPluginManager $plugin_manager
   *   The LightGallery plugin manager.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    ExtensionPathResolver $extension_path_resolver,
    ModuleExtensionList $module_extension_list,
    string $app_root,
    ModuleHandlerInterface $module_handler,
    LightgalleryPluginManager $plugin_manager,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->extensionPathResolver = $extension_path_resolver;
    $this->moduleExtensionList = $module_extension_list;
    $this->appRoot = $app_root;
    $this->moduleHandler = $module_handler;
    $this->pluginManager = $plugin_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    // @phpstan-ignore-next-line new.static is standard Drupal DI pattern.
    $instance = new static(
      $container->get('entity_type.manager'),
      $container->get('extension.path.resolver'),
      $container->get('extension.list.module'),
      $container->getParameter('app.root'),
      $container->get('module_handler'),
      $container->get('plugin.manager.lightgallery')
    );

    // Inject preview builder if the preview module is enabled.
    if ($container->get('module_handler')->moduleExists('lightgallery_formatter_preview')) {
      $instance->previewBuilder = $container->get('lightgallery_formatter_preview.preview_builder');
    }

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function init(FormStateInterface $form_state) {
    // Ensure $this->entity is set before calling parent::init().
    if (!$this->entity) {
      $storage = $this->entityTypeManager->getStorage('lightgallery_profile');
      $this->entity = $storage->create([
        'status' => TRUE,
      ]);
    }

    // Now safe to call parent.
    parent::init($form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface $profile */
    $profile = $this->entity;

    // Preview section at the top of the form.
    $form['preview'] = [
      '#type' => 'details',
      '#title' => $this->t('Gallery Preview'),
      '#open' => TRUE,
      '#weight' => -100,
      '#attributes' => [
        'id' => 'lightgallery-preview-wrapper',
        'class' => ['lightgallery-preview-section'],
      ],
    ];

    $form['preview']['description'] = [
      '#type' => 'markup',
      '#markup' => '<p class="lightgallery-preview-description">' . $this->t('Click "Generate Preview" to see how the gallery will look with current settings.') . '</p>',
    ];

    $form['preview']['generate_button'] = [
      '#type' => 'button',
      '#value' => $this->t('Generate Preview'),
      '#ajax' => [
        'callback' => '::generatePreview',
        'wrapper' => 'lightgallery-preview-content',
        'effect' => 'fade',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Generating preview...'),
        ],
      ],
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
    ];

    $form['preview']['content'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'lightgallery-preview-content',
        'class' => ['lightgallery-preview-content'],
      ],
    ];

    // Check if we need to show the preview (after AJAX callback).
    if ($form_state->get('show_preview')) {
      $form['preview']['content']['gallery'] = $this->buildPreviewGallery($form_state);
    }
    else {
      $form['preview']['content']['placeholder'] = [
        '#type' => 'markup',
        '#markup' => '<div class="lightgallery-preview-placeholder">' . $this->t('Preview will appear here after clicking "Generate Preview".') . '</div>',
      ];
    }

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#description' => $this->t('The human-readable name for this profile.'),
      '#default_value' => $profile->label(),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#title' => $this->t('Machine name'),
      '#description' => $this->t('A unique name for this profile. Can only contain lowercase letters, numbers, and underscores.'),
      '#default_value' => $profile->id(),
      '#disabled' => !$profile->isNew(),
      '#machine_name' => [
        'exists' => '\Drupal\lightgallery_formatter\Entity\LightgalleryProfile::load',
        'source' => ['label'],
      ],
      '#required' => TRUE,
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $profile->getStatus(),
    ];

    // Build plugin forms dynamically.
    // Each plugin form is added directly to the form with #tree => TRUE
    // to ensure form values are properly nested by plugin_id.
    foreach ($this->pluginManager->getSortedDefinitions() as $plugin_id => $definition) {
      $plugin_settings = $profile->getPluginSettings($plugin_id);
      $plugin = $this->pluginManager->createInstance($plugin_id, $plugin_settings);
      $plugin_form = $plugin->buildForm($form, $form_state, $profile);
      // Ensure the plugin form has #tree enabled for proper nesting.
      $plugin_form['#tree'] = TRUE;
      $form[$plugin_id] = $plugin_form;
    }

    return $form;
  }

  /**
   * AJAX callback to generate the gallery preview.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The preview content render array.
   */
  public function generatePreview(array &$form, FormStateInterface $form_state) {
    // Set flag to show preview.
    $form_state->set('show_preview', TRUE);

    // Rebuild the preview content.
    $form['preview']['content']['gallery'] = $this->buildPreviewGallery($form_state);

    // Remove placeholder if exists.
    unset($form['preview']['content']['placeholder']);

    return $form['preview']['content'];
  }

  /**
   * Build the preview gallery render array.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The gallery render array.
   */
  protected function buildPreviewGallery(FormStateInterface $form_state) {
    // Check if preview module is enabled.
    if (!$this->moduleHandler->moduleExists('lightgallery_formatter_preview')) {
      return [
        '#type' => 'container',
        '#attributes' => ['class' => ['lightgallery-preview-not-available']],
        'message' => [
          '#markup' => '<div class="messages messages--warning">' .
          $this->t('The LightGallery Formatter Preview module is not enabled. Please enable it to use the preview functionality.') .
          '</div>',
        ],
      ];
    }

    // Build settings from form state.
    $settings = $this->buildLightgallerySettingsFromForm($form_state);

    // Use preview builder service.
    if ($this->previewBuilder) {
      return $this->previewBuilder->buildPreview($settings);
    }

    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['lightgallery-preview-error']],
      'message' => [
        '#markup' => '<div class="messages messages--error">' .
        $this->t('Preview service is not available.') .
        '</div>',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate each plugin's form.
    foreach ($this->pluginManager->getSortedDefinitions() as $plugin_id => $definition) {
      $plugin = $this->pluginManager->createInstance($plugin_id);
      $plugin->validateForm($form, $form_state);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    /** @var \Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface $profile */
    $profile = $this->entity;

    // Submit each plugin's form and save settings.
    foreach ($this->pluginManager->getSortedDefinitions() as $plugin_id => $definition) {
      $plugin = $this->pluginManager->createInstance($plugin_id);
      $settings = $plugin->submitForm($form, $form_state);
      $profile->setPluginSettings($plugin_id, $settings);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $profile = $this->entity;
    $status = $profile->save();

    if ($status === SAVED_NEW) {
      $this->messenger()->addStatus($this->t('Created the %label LightGallery profile.', [
        '%label' => $profile->label(),
      ]));
    }
    else {
      $this->messenger()->addStatus($this->t('Saved the %label LightGallery profile.', [
        '%label' => $profile->label(),
      ]));
    }

    $form_state->setRedirectUrl($profile->toUrl('collection'));

    return $status;
  }

  /**
   * Build LightGallery settings from form state.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   LightGallery settings array.
   */
  protected function buildLightgallerySettingsFromForm(FormStateInterface $form_state) {
    /** @var \Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface $profile */
    $profile = $this->entity;

    // Build settings from all enabled plugins.
    $settings = ['selector' => 'a'];

    foreach ($this->pluginManager->getSortedDefinitions() as $plugin_id => $definition) {
      // Get current form values or fall back to profile settings.
      $form_values = $form_state->getValue($plugin_id);
      if ($form_values !== NULL) {
        // Create a temporary form state to use plugin's submitForm.
        // Note: submitForm expects $form by reference, so we need a variable.
        $temp_form = [];
        $temp_form_state = clone $form_state;
        $plugin = $this->pluginManager->createInstance($plugin_id);
        $plugin_settings = $plugin->submitForm($temp_form, $temp_form_state);
      }
      else {
        $plugin_settings = $profile->getPluginSettings($plugin_id);
      }

      // Build JS settings using the plugin.
      $plugin = $this->pluginManager->createInstance($plugin_id, $plugin_settings);
      if ($plugin->isEnabled($plugin_settings)) {
        $settings = array_merge($settings, $plugin->buildJsSettings($plugin_settings));
      }
    }

    return $settings;
  }

}
