<?php

namespace Drupal\lightgallery_formatter\Plugin\Lightgallery;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\lightgallery_formatter\Attribute\LightgalleryPlugin;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface;
use Drupal\lightgallery_formatter\LightgalleryPluginBase;

/**
 * Provides general LightGallery settings plugin.
 */
#[LightgalleryPlugin(
  id: 'general',
  label: new TranslatableMarkup('General Settings'),
  description: new TranslatableMarkup('Core LightGallery configuration options including transitions, navigation, and controls.'),
  weight: -100,
)]
class General extends LightgalleryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultSettings(): array {
    return [
      'licenseKey' => '0000-0000-000-0000',
      'loop' => TRUE,
      'preload' => 2,
      'speed' => 400,
      'mode' => 'lg-slide',
      'closable' => TRUE,
      'closeOnTap' => TRUE,
      'escKey' => TRUE,
      'keyPress' => TRUE,
      'showCloseIcon' => TRUE,
      'showMaximizeIcon' => FALSE,
      'hideBarsDelay' => 0,
      'counter' => TRUE,
      'download' => TRUE,
      'enableDrag' => TRUE,
      'enableSwipe' => TRUE,
      'mousewheel' => FALSE,
      'allowMediaOverlap' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(array $settings): bool {
    // General plugin is always enabled.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array &$form, FormStateInterface $form_state, LightgalleryProfileInterface $profile): array {
    $settings = $this->mergeWithDefaults($profile->getPluginSettings('general'));

    $element = [
      '#type' => 'details',
      '#title' => $this->t('General Settings'),
      '#open' => TRUE,
    ];

    $element['licenseKey'] = [
      '#type' => 'textfield',
      '#title' => $this->t('License key'),
      '#default_value' => $settings['licenseKey'],
      '#description' => $this->t('Your LightGallery license key. Get one at <a href="https://www.lightgalleryjs.com/license/" target="_blank">lightgalleryjs.com/license</a>. Use "0000-0000-000-0000" for non-commercial use.'),
      '#placeholder' => '0000-0000-000-0000',
    ];

    $element['loop'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable looping'),
      '#default_value' => $settings['loop'],
      '#description' => $this->t('Allow the gallery to loop from last to first image.'),
    ];

    $element['preload'] = [
      '#type' => 'number',
      '#title' => $this->t('Preload count'),
      '#default_value' => $settings['preload'],
      '#min' => 1,
      '#max' => 5,
      '#description' => $this->t('Number of images to load in advance.'),
    ];

    $element['speed'] = [
      '#type' => 'number',
      '#title' => $this->t('Transition speed (ms)'),
      '#default_value' => $settings['speed'],
      '#min' => 0,
      '#description' => $this->t('Duration of transition between images in milliseconds.'),
    ];

    $element['mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Transition mode'),
      '#default_value' => $settings['mode'],
      '#options' => $this->getTransitionModes(),
      '#description' => $this->t('Type of transition between images.'),
    ];

    $element['closable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Closable'),
      '#default_value' => $settings['closable'],
      '#description' => $this->t("If disabled, users won't be able to close the gallery. Useful for inline galleries."),
    ];

    $element['closeOnTap'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Close on tap'),
      '#default_value' => $settings['closeOnTap'],
      '#description' => $this->t('Allows clicks on black area to close gallery.'),
    ];

    $element['escKey'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Close with Esc key'),
      '#default_value' => $settings['escKey'],
      '#description' => $this->t('Whether the gallery can be closed by pressing the "Esc" key.'),
    ];

    $element['keyPress'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Keyboard navigation'),
      '#default_value' => $settings['keyPress'],
      '#description' => $this->t('Enable keyboard navigation (arrow keys, etc.).'),
    ];

    $element['showCloseIcon'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show close icon'),
      '#default_value' => $settings['showCloseIcon'],
      '#description' => $this->t('Display the close button in the gallery.'),
    ];

    $element['showMaximizeIcon'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show maximize icon'),
      '#default_value' => $settings['showMaximizeIcon'],
      '#description' => $this->t('Display the maximize button in the gallery.'),
    ];

    $element['hideBarsDelay'] = [
      '#type' => 'number',
      '#title' => $this->t('Hide controls delay (ms)'),
      '#default_value' => $settings['hideBarsDelay'],
      '#min' => 0,
      '#description' => $this->t('Delay in milliseconds before hiding gallery controls after interaction. Set to 0 to keep controls visible.'),
    ];

    $element['counter'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show counter'),
      '#default_value' => $settings['counter'],
      '#description' => $this->t('Display the total number of images and index of the currently displayed image.'),
    ];

    $element['download'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable download'),
      '#default_value' => $settings['download'],
      '#description' => $this->t('Display the download button for images.'),
    ];

    $element['enableDrag'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable desktop drag'),
      '#default_value' => $settings['enableDrag'],
      '#description' => $this->t('Enable mouse drag support for desktop devices.'),
    ];

    $element['enableSwipe'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable touch swipe'),
      '#default_value' => $settings['enableSwipe'],
      '#description' => $this->t('Enable swipe support for touch devices.'),
    ];

    $element['mousewheel'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Mouse wheel navigation'),
      '#default_value' => $settings['mousewheel'],
      '#description' => $this->t('Enable navigation to next/prev slides using mouse wheel.'),
    ];

    $element['allowMediaOverlap'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow media overlap'),
      '#default_value' => $settings['allowMediaOverlap'],
      '#description' => $this->t('Allow thumbnails and captions to overlay the main image. <strong>Recommended: Keep enabled</strong> to show the "Toggle thumbnails" button and other plugin icons. When disabled, thumbnails appear in a fixed area below the image.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $plugin_id = $this->getPluginId();

    $preload = $form_state->getValue([$plugin_id, 'preload']);
    if ($preload !== NULL && (!is_numeric($preload) || $preload < 0)) {
      $form_state->setErrorByName("{$plugin_id}][preload", $this->t('Preload count must be a positive number.'));
    }

    $speed = $form_state->getValue([$plugin_id, 'speed']);
    if ($speed !== NULL && (!is_numeric($speed) || $speed < 0)) {
      $form_state->setErrorByName("{$plugin_id}][speed", $this->t('Transition speed must be a positive number.'));
    }

    $hideBarsDelay = $form_state->getValue([$plugin_id, 'hideBarsDelay']);
    if ($hideBarsDelay !== NULL && (!is_numeric($hideBarsDelay) || $hideBarsDelay < 0)) {
      $form_state->setErrorByName("{$plugin_id}][hideBarsDelay", $this->t('Hide controls delay must be a positive number.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): array {
    $plugin_id = $this->getPluginId();
    $values = $form_state->getValue($plugin_id) ?? [];

    return [
      'licenseKey' => $values['licenseKey'] ?? '0000-0000-000-0000',
      'loop' => (bool) ($values['loop'] ?? TRUE),
      'preload' => (int) ($values['preload'] ?? 2),
      'speed' => (int) ($values['speed'] ?? 400),
      'mode' => $values['mode'] ?? 'lg-slide',
      'closable' => (bool) ($values['closable'] ?? TRUE),
      'closeOnTap' => (bool) ($values['closeOnTap'] ?? TRUE),
      'escKey' => (bool) ($values['escKey'] ?? TRUE),
      'keyPress' => (bool) ($values['keyPress'] ?? TRUE),
      'showCloseIcon' => (bool) ($values['showCloseIcon'] ?? TRUE),
      'showMaximizeIcon' => (bool) ($values['showMaximizeIcon'] ?? FALSE),
      'hideBarsDelay' => (int) ($values['hideBarsDelay'] ?? 0),
      'counter' => (bool) ($values['counter'] ?? TRUE),
      'download' => (bool) ($values['download'] ?? TRUE),
      'enableDrag' => (bool) ($values['enableDrag'] ?? TRUE),
      'enableSwipe' => (bool) ($values['enableSwipe'] ?? TRUE),
      'mousewheel' => (bool) ($values['mousewheel'] ?? FALSE),
      'allowMediaOverlap' => (bool) ($values['allowMediaOverlap'] ?? TRUE),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildJsSettings(array $settings): array {
    $merged = $this->mergeWithDefaults($settings);

    return [
      'licenseKey' => $merged['licenseKey'],
      'loop' => (bool) $merged['loop'],
      'preload' => (int) $merged['preload'],
      'speed' => (int) $merged['speed'],
      'mode' => $merged['mode'],
      'closable' => (bool) $merged['closable'],
      'closeOnTap' => (bool) $merged['closeOnTap'],
      'escKey' => (bool) $merged['escKey'],
      'keyPress' => (bool) $merged['keyPress'],
      'showCloseIcon' => (bool) $merged['showCloseIcon'],
      'showMaximizeIcon' => (bool) $merged['showMaximizeIcon'],
      'hideBarsDelay' => (int) $merged['hideBarsDelay'],
      'counter' => (bool) $merged['counter'],
      'download' => (bool) $merged['download'],
      'enableDrag' => (bool) $merged['enableDrag'],
      'enableSwipe' => (bool) $merged['enableSwipe'],
      'mousewheel' => (bool) $merged['mousewheel'],
      'allowMediaOverlap' => (bool) $merged['allowMediaOverlap'],
    ];
  }

  /**
   * Gets available transition modes.
   *
   * @return array
   *   An array of transition mode options.
   */
  protected function getTransitionModes(): array {
    return [
      'lg-slide' => $this->t('Slide'),
      'lg-fade' => $this->t('Fade'),
      'lg-zoom-in' => $this->t('Zoom in'),
      'lg-zoom-in-big' => $this->t('Zoom in big'),
      'lg-zoom-out' => $this->t('Zoom out'),
      'lg-zoom-out-big' => $this->t('Zoom out big'),
      'lg-zoom-out-in' => $this->t('Zoom out in'),
      'lg-zoom-in-out' => $this->t('Zoom in out'),
      'lg-soft-zoom' => $this->t('Soft zoom'),
      'lg-scale-up' => $this->t('Scale up'),
      'lg-slide-circular' => $this->t('Slide circular'),
      'lg-slide-circular-vertical' => $this->t('Slide circular vertical'),
      'lg-slide-vertical' => $this->t('Slide vertical'),
      'lg-slide-vertical-growth' => $this->t('Slide vertical growth'),
      'lg-slide-skew-only' => $this->t('Slide skew only'),
      'lg-slide-skew-only-rev' => $this->t('Slide skew only reverse'),
      'lg-slide-skew-only-y' => $this->t('Slide skew only Y'),
      'lg-slide-skew-only-y-rev' => $this->t('Slide skew only Y reverse'),
      'lg-slide-skew' => $this->t('Slide skew'),
      'lg-slide-skew-rev' => $this->t('Slide skew reverse'),
      'lg-slide-skew-cross' => $this->t('Slide skew cross'),
      'lg-slide-skew-cross-rev' => $this->t('Slide skew cross reverse'),
      'lg-slide-skew-ver' => $this->t('Slide skew vertical'),
      'lg-slide-skew-ver-rev' => $this->t('Slide skew vertical reverse'),
      'lg-slide-skew-ver-cross' => $this->t('Slide skew vertical cross'),
      'lg-slide-skew-ver-cross-rev' => $this->t('Slide skew vertical cross reverse'),
      'lg-lollipop' => $this->t('Lollipop'),
      'lg-lollipop-rev' => $this->t('Lollipop reverse'),
      'lg-rotate' => $this->t('Rotate'),
      'lg-rotate-rev' => $this->t('Rotate reverse'),
      'lg-tube' => $this->t('Tube'),
    ];
  }

}
