<?php

namespace Drupal\lightgallery_formatter\Plugin\Lightgallery;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\lightgallery_formatter\Attribute\LightgalleryPlugin;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface;
use Drupal\lightgallery_formatter\LightgalleryPluginBase;

/**
 * Provides hash plugin for LightGallery.
 *
 * The galleryId is automatically generated by the field formatter using
 * Drupal's unique ID system to ensure consistency between the container
 * element and the URL hash.
 */
#[LightgalleryPlugin(
  id: 'hash',
  label: new TranslatableMarkup('Hash'),
  description: new TranslatableMarkup('Enable URL hash navigation for the gallery.'),
  weight: 55,
)]
class Hash extends LightgalleryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultSettings(): array {
    return [
      'enabled' => FALSE,
      'hash' => TRUE,
      'customSlideName' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(array $settings): bool {
    return !empty($settings['enabled']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array &$form, FormStateInterface $form_state, LightgalleryProfileInterface $profile): array {
    $settings = $this->mergeWithDefaults($profile->getPluginSettings('hash'));
    $plugin_id = $this->getPluginId();

    $element = [
      '#type' => 'details',
      '#title' => $this->t('Hash'),
      '#open' => !empty($settings['enabled']),
    ];

    $element['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable hash plugin'),
      '#default_value' => $settings['enabled'],
      '#description' => $this->t('Enable URL hash navigation. Allows sharing direct links to specific images.'),
    ];

    $element['settings_wrapper'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="' . $plugin_id . '[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Note about auto-generated galleryId.
    $element['settings_wrapper']['gallery_id_note'] = [
      '#type' => 'item',
      '#markup' => $this->t('<strong>Note:</strong> The Gallery ID is automatically generated by Drupal based on the field name and entity. This ensures each gallery on the page has a unique identifier in the URL hash, preventing conflicts when multiple galleries exist on the same page.'),
      '#prefix' => '<div class="messages messages--status">',
      '#suffix' => '</div>',
    ];

    $element['settings_wrapper']['hash'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable hash option'),
      '#default_value' => $settings['hash'],
      '#description' => $this->t('Enable/Disable hash in URL when navigating gallery.'),
    ];

    $element['settings_wrapper']['customSlideName'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Custom slide name'),
      '#default_value' => $settings['customSlideName'],
      '#description' => $this->t('Use custom slide names in the URL instead of slide index.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // No validation needed - galleryId is auto-generated by the formatter.
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): array {
    $plugin_id = $this->getPluginId();
    $values = $form_state->getValue($plugin_id) ?? [];

    if (empty($values['enabled'])) {
      return ['enabled' => FALSE];
    }

    $settings_wrapper = $values['settings_wrapper'] ?? [];

    return [
      'enabled' => TRUE,
      'hash' => (bool) ($settings_wrapper['hash'] ?? TRUE),
      'customSlideName' => (bool) ($settings_wrapper['customSlideName'] ?? FALSE),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildJsSettings(array $settings): array {
    if (!$this->isEnabled($settings)) {
      return [];
    }

    $merged = $this->mergeWithDefaults($settings);

    // Note: galleryId is set by the field formatter using Drupal's unique ID.
    // This ensures the hash in the URL matches the actual gallery container.
    return [
      'hash' => (bool) $merged['hash'],
      'customSlideName' => (bool) $merged['customSlideName'],
    ];
  }

}
