<?php

namespace Drupal\lightgallery_formatter\Plugin\Lightgallery;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\lightgallery_formatter\Attribute\LightgalleryPlugin;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface;
use Drupal\lightgallery_formatter\LightgalleryPluginBase;

/**
 * Provides relative caption plugin for LightGallery.
 */
#[LightgalleryPlugin(
  id: 'relative_caption',
  label: new TranslatableMarkup('Relative Caption'),
  description: new TranslatableMarkup('Display captions relative to the image position instead of fixed at bottom.'),
  weight: 30,
)]
class RelativeCaption extends LightgalleryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultSettings(): array {
    return [
      'enabled' => FALSE,
      'relativeCaption' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(array $settings): bool {
    return !empty($settings['enabled']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array &$form, FormStateInterface $form_state, LightgalleryProfileInterface $profile): array {
    $settings = $this->mergeWithDefaults(
      $profile->getPluginSettings('relative_caption')
    );
    $plugin_id = $this->getPluginId();

    $element = [
      '#type' => 'details',
      '#title' => $this->t('Relative Caption'),
      '#open' => !empty($settings['enabled']),
    ];

    $element['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable relative caption plugin'),
      '#default_value' => $settings['enabled'],
      '#description' => $this->t('Display captions relative to the image position.'),
    ];

    $element['settings_wrapper'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="' . $plugin_id . '[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $element['settings_wrapper']['relativeCaption'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable relative captions'),
      '#default_value' => $settings['relativeCaption'],
      '#description' => $this->t('When enabled, captions appear below the image instead of at the bottom of the screen.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): array {
    $plugin_id = $this->getPluginId();
    $values = $form_state->getValue($plugin_id) ?? [];

    if (empty($values['enabled'])) {
      return ['enabled' => FALSE];
    }

    $settings_wrapper = $values['settings_wrapper'] ?? [];

    return [
      'enabled' => TRUE,
      'relativeCaption' => (bool) ($settings_wrapper['relativeCaption'] ?? FALSE),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildJsSettings(array $settings): array {
    if (!$this->isEnabled($settings)) {
      return [];
    }

    $merged = $this->mergeWithDefaults($settings);

    return [
      'relativeCaption' => (bool) $merged['relativeCaption'],
    ];
  }

}
