<?php

namespace Drupal\lightgallery_formatter\Plugin\Lightgallery;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\lightgallery_formatter\Attribute\LightgalleryPlugin;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface;
use Drupal\lightgallery_formatter\LightgalleryPluginBase;

/**
 * Provides rotate plugin for LightGallery.
 */
#[LightgalleryPlugin(
  id: 'rotate',
  label: new TranslatableMarkup('Rotate'),
  description: new TranslatableMarkup('Enable rotation and flip options for gallery images.'),
  weight: 35,
)]
class Rotate extends LightgalleryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultSettings(): array {
    return [
      'enabled' => FALSE,
      'rotate' => TRUE,
      'rotateSpeed' => 400,
      'rotateLeft' => TRUE,
      'rotateRight' => TRUE,
      'flipHorizontal' => TRUE,
      'flipVertical' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(array $settings): bool {
    return !empty($settings['enabled']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array &$form, FormStateInterface $form_state, LightgalleryProfileInterface $profile): array {
    $settings = $this->mergeWithDefaults($profile->getPluginSettings('rotate'));
    $plugin_id = $this->getPluginId();

    $element = [
      '#type' => 'details',
      '#title' => $this->t('Rotate'),
      '#open' => !empty($settings['enabled']),
    ];

    $element['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable rotate plugin'),
      '#default_value' => $settings['enabled'],
      '#description' => $this->t('Allow users to rotate and flip gallery images.'),
    ];

    $element['settings_wrapper'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="' . $plugin_id . '[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $element['settings_wrapper']['rotate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable rotate option'),
      '#default_value' => $settings['rotate'],
      '#description' => $this->t('Enable/Disable rotate functionality.'),
    ];

    $element['settings_wrapper']['rotateSpeed'] = [
      '#type' => 'number',
      '#title' => $this->t('Rotate speed'),
      '#default_value' => $settings['rotateSpeed'],
      '#min' => 0,
      '#max' => 2000,
      '#step' => 50,
      '#field_suffix' => 'ms',
      '#description' => $this->t('Rotate animation speed in milliseconds.'),
    ];

    $element['settings_wrapper']['rotateLeft'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable rotate left'),
      '#default_value' => $settings['rotateLeft'],
      '#description' => $this->t('Show rotate left button.'),
    ];

    $element['settings_wrapper']['rotateRight'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable rotate right'),
      '#default_value' => $settings['rotateRight'],
      '#description' => $this->t('Show rotate right button.'),
    ];

    $element['settings_wrapper']['flipHorizontal'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable flip horizontal'),
      '#default_value' => $settings['flipHorizontal'],
      '#description' => $this->t('Show flip horizontal button.'),
    ];

    $element['settings_wrapper']['flipVertical'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable flip vertical'),
      '#default_value' => $settings['flipVertical'],
      '#description' => $this->t('Show flip vertical button.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): array {
    $plugin_id = $this->getPluginId();
    $values = $form_state->getValue($plugin_id) ?? [];

    if (empty($values['enabled'])) {
      return ['enabled' => FALSE];
    }

    $settings_wrapper = $values['settings_wrapper'] ?? [];

    return [
      'enabled' => TRUE,
      'rotate' => (bool) ($settings_wrapper['rotate'] ?? TRUE),
      'rotateSpeed' => (int) ($settings_wrapper['rotateSpeed'] ?? 400),
      'rotateLeft' => (bool) ($settings_wrapper['rotateLeft'] ?? TRUE),
      'rotateRight' => (bool) ($settings_wrapper['rotateRight'] ?? TRUE),
      'flipHorizontal' => (bool) ($settings_wrapper['flipHorizontal'] ?? TRUE),
      'flipVertical' => (bool) ($settings_wrapper['flipVertical'] ?? TRUE),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildJsSettings(array $settings): array {
    if (!$this->isEnabled($settings)) {
      return [];
    }

    $merged = $this->mergeWithDefaults($settings);

    return [
      'rotate' => (bool) $merged['rotate'],
      'rotateSpeed' => (int) $merged['rotateSpeed'],
      'rotateLeft' => (bool) $merged['rotateLeft'],
      'rotateRight' => (bool) $merged['rotateRight'],
      'flipHorizontal' => (bool) $merged['flipHorizontal'],
      'flipVertical' => (bool) $merged['flipVertical'],
    ];
  }

}
