<?php

namespace Drupal\lightgallery_formatter\Plugin\Lightgallery;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\lightgallery_formatter\Attribute\LightgalleryPlugin;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface;
use Drupal\lightgallery_formatter\LightgalleryPluginBase;

/**
 * Provides zoom functionality plugin for LightGallery.
 */
#[LightgalleryPlugin(
  id: 'zoom',
  label: new TranslatableMarkup('Zoom'),
  description: new TranslatableMarkup('Enable zoom functionality for gallery images.'),
  weight: 10,
)]
class Zoom extends LightgalleryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultSettings(): array {
    return [
      'enabled' => FALSE,
      'scale' => 1,
      'infiniteZoom' => TRUE,
      'actualSize' => TRUE,
      'showZoomInOutIcons' => FALSE,
      'enableZoomAfter' => 300,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(array $settings): bool {
    return !empty($settings['enabled']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(
    array &$form,
    FormStateInterface $form_state,
    LightgalleryProfileInterface $profile,
  ): array {
    $settings = $this->mergeWithDefaults($profile->getPluginSettings('zoom'));
    $plugin_id = $this->getPluginId();

    $element = [
      '#type' => 'details',
      '#title' => $this->t('Zoom'),
      '#open' => !empty($settings['enabled']),
    ];

    $element['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable zoom plugin'),
      '#default_value' => $settings['enabled'],
      '#description' => $this->t('Allow users to zoom in/out on gallery images.'),
    ];

    $element['settings_wrapper'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="' . $plugin_id . '[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $element['settings_wrapper']['scale'] = [
      '#type' => 'number',
      '#title' => $this->t('Zoom scale'),
      '#default_value' => $settings['scale'],
      '#min' => 0.1,
      '#max' => 5,
      '#step' => 0.1,
      '#description' => $this->t('Value by which zoom should be incremented/decremented.'),
    ];

    $element['settings_wrapper']['infiniteZoom'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Infinite zoom'),
      '#default_value' => $settings['infiniteZoom'],
      '#description' => $this->t('Allow zooming beyond the original image size.'),
    ];

    $element['settings_wrapper']['actualSize'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show actual size button'),
      '#default_value' => $settings['actualSize'],
      '#description' => $this->t('Display a button to view the image at its actual size.'),
    ];

    $element['settings_wrapper']['showZoomInOutIcons'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show zoom in/out icons'),
      '#default_value' => $settings['showZoomInOutIcons'],
      '#description' => $this->t('Display zoom in (+) and zoom out (-) icons.'),
    ];

    $element['settings_wrapper']['enableZoomAfter'] = [
      '#type' => 'number',
      '#title' => $this->t('Enable zoom after'),
      '#default_value' => $settings['enableZoomAfter'],
      '#min' => 0,
      '#max' => 5000,
      '#field_suffix' => 'ms',
      '#description' => $this->t('Delay before activating zoom after slide transition completes.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $plugin_id = $this->getPluginId();

    $scale = $form_state->getValue([$plugin_id, 'settings_wrapper', 'scale']);
    if ($scale !== NULL && (!is_numeric($scale) || $scale < 0.1)) {
      $form_state->setErrorByName(
        "{$plugin_id}][settings_wrapper][scale",
        $this->t('Zoom scale must be at least 0.1.')
      );
    }

    $enableZoomAfter = $form_state->getValue([
      $plugin_id,
      'settings_wrapper',
      'enableZoomAfter',
    ]);
    if ($enableZoomAfter !== NULL) {
      if (!is_numeric($enableZoomAfter) || $enableZoomAfter < 0) {
        $form_state->setErrorByName(
          "{$plugin_id}][settings_wrapper][enableZoomAfter",
          $this->t('Enable zoom after must be a positive number.')
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): array {
    $plugin_id = $this->getPluginId();
    $values = $form_state->getValue($plugin_id) ?? [];

    // If not enabled, return minimal settings.
    if (empty($values['enabled'])) {
      return ['enabled' => FALSE];
    }

    $settings_wrapper = $values['settings_wrapper'] ?? [];

    return [
      'enabled' => TRUE,
      'scale' => (float) ($settings_wrapper['scale'] ?? 1),
      'infiniteZoom' => (bool) ($settings_wrapper['infiniteZoom'] ?? TRUE),
      'actualSize' => (bool) ($settings_wrapper['actualSize'] ?? TRUE),
      'showZoomInOutIcons' => (bool) ($settings_wrapper['showZoomInOutIcons'] ?? FALSE),
      'enableZoomAfter' => (int) ($settings_wrapper['enableZoomAfter'] ?? 300),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildJsSettings(array $settings): array {
    // Only return settings if enabled.
    if (!$this->isEnabled($settings)) {
      return [];
    }

    $merged = $this->mergeWithDefaults($settings);

    return [
      'zoom' => TRUE,
      'scale' => (float) $merged['scale'],
      'infiniteZoom' => (bool) $merged['infiniteZoom'],
      'actualSize' => (bool) $merged['actualSize'],
      'showZoomInOutIcons' => (bool) $merged['showZoomInOutIcons'],
      'enableZoomAfter' => (int) $merged['enableZoomAfter'],
    ];
  }

}
