<?php

namespace Drupal\Tests\lightgallery_formatter\Functional;

use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests access control for LightGallery profile admin pages.
 *
 * @group lightgallery_formatter
 */
class LightgalleryProfileAccessTest extends BrowserTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * The default theme.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * An admin user with permission.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A regular user without permission.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $regularUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'administer lightgallery_formatter',
      'administer site configuration',
    ]);

    $this->regularUser = $this->drupalCreateUser([
      'access content',
    ]);
  }

  /**
   * Test admin can access profile list.
   */
  public function testAdminCanAccessProfileList(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/structure/lightgallery-profiles');

    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('LightGallery');
  }

  /**
   * Test admin can create profile.
   */
  public function testAdminCanCreateProfile(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/structure/lightgallery-profiles/add');

    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->fieldExists('label');
    $this->assertSession()->fieldExists('id');

    // Submit the form.
    $edit = [
      'label' => 'Test Profile',
      'id' => 'test_profile',
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->pageTextContains('Created the Test Profile');

    // Verify profile exists.
    $profile = LightgalleryProfile::load('test_profile');
    $this->assertNotNull($profile, 'Profile was created.');
  }

  /**
   * Test admin can edit profile.
   */
  public function testAdminCanEditProfile(): void {
    // Create a profile first.
    $profile = LightgalleryProfile::create([
      'id' => 'edit_test',
      'label' => 'Edit Test',
      'status' => TRUE,
    ]);
    $profile->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/structure/lightgallery-profiles/edit_test');

    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->fieldValueEquals('label', 'Edit Test');

    // Update the profile.
    $edit = [
      'label' => 'Updated Profile',
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->pageTextContains('Saved the Updated Profile');
  }

  /**
   * Test admin can delete profile.
   */
  public function testAdminCanDeleteProfile(): void {
    // Create a profile to delete.
    $profile = LightgalleryProfile::create([
      'id' => 'delete_test',
      'label' => 'Delete Test',
      'status' => TRUE,
    ]);
    $profile->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/structure/lightgallery-profiles/delete_test/delete');

    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Are you sure');

    // Confirm deletion.
    $this->submitForm([], 'Delete');

    // Verify profile is deleted.
    $loaded = LightgalleryProfile::load('delete_test');
    $this->assertNull($loaded, 'Profile was deleted.');
  }

  /**
   * Test anonymous cannot access profile list.
   */
  public function testAnonymousCannotAccessProfileList(): void {
    $this->drupalGet('/admin/structure/lightgallery-profiles');

    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Test anonymous cannot access add form.
   */
  public function testAnonymousCannotAccessAddForm(): void {
    $this->drupalGet('/admin/structure/lightgallery-profiles/add');

    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Test anonymous cannot access edit form.
   */
  public function testAnonymousCannotAccessEditForm(): void {
    $this->drupalGet('/admin/structure/lightgallery-profiles/default');

    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Test regular user without permission denied access.
   */
  public function testUserWithoutPermissionDenied(): void {
    $this->drupalLogin($this->regularUser);

    // Try to access list.
    $this->drupalGet('/admin/structure/lightgallery-profiles');
    $this->assertSession()->statusCodeEquals(403);

    // Try to access add.
    $this->drupalGet('/admin/structure/lightgallery-profiles/add');
    $this->assertSession()->statusCodeEquals(403);

    // Try to access edit.
    $this->drupalGet('/admin/structure/lightgallery-profiles/default');
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Test that only admin permission grants access.
   */
  public function testOnlyAdminPermissionGrantsAccess(): void {
    // Create user with only site config permission.
    $siteConfigUser = $this->drupalCreateUser([
      'administer site configuration',
    ]);

    $this->drupalLogin($siteConfigUser);
    $this->drupalGet('/admin/structure/lightgallery-profiles');

    // Denied - missing lightgallery_formatter permission.
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Test profile delete page shows correct profile name.
   */
  public function testDeletePageShowsProfileName(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'named_profile',
      'label' => 'My Named Profile',
      'status' => TRUE,
    ]);
    $profile->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/structure/lightgallery-profiles/named_profile/delete');

    $this->assertSession()->pageTextContains('My Named Profile');
  }

}
