<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\KernelTests\KernelTestBase;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;

/**
 * Tests cacheability metadata for LightGallery profiles.
 *
 * @group lightgallery_formatter
 */
class LightgalleryCacheabilityTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
  }

  /**
   * Test profile entity has cache tags.
   */
  public function testProfileEntityHasCacheTags(): void {
    $profile = LightgalleryProfile::load('default');

    $cache_tags = $profile->getCacheTags();

    $this->assertNotEmpty($cache_tags, 'Profile has cache tags.');
    $this->assertContains('config:lightgallery_formatter.profile.default', $cache_tags, 'Profile has specific config cache tag.');
  }

  /**
   * Test profile cache tag format.
   */
  public function testProfileCacheTagFormat(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'cache_test',
      'label' => 'Cache Test',
      'status' => TRUE,
    ]);
    $profile->save();

    $cache_tags = $profile->getCacheTags();

    $this->assertContains(
      'config:lightgallery_formatter.profile.cache_test',
      $cache_tags,
      'Cache tag follows expected format.'
    );
  }

  /**
   * Test that modifying profile invalidates cache.
   */
  public function testModifyingProfileInvalidatesCache(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'invalidate_test',
      'label' => 'Invalidate Test',
      'status' => TRUE,
    ]);
    $profile->save();

    // Get initial cache tags.
    $initial_tags = $profile->getCacheTags();

    // Modify and save.
    $profile->set('label', 'Updated Invalidate Test');
    $profile->save();

    // Reload and verify tags are same (but cache would be invalidated).
    $reloaded = LightgalleryProfile::load('invalidate_test');
    $updated_tags = $reloaded->getCacheTags();

    $this->assertEquals($initial_tags, $updated_tags, 'Cache tags are consistent.');

    // The important thing is that saving triggers cache invalidation.
    // We can't easily test invalidation directly in kernel tests,
    // but we verify the tags are correctly formed.
  }

  /**
   * Test config entity list cache tags.
   */
  public function testConfigEntityListCacheTags(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'list_tag_test',
      'label' => 'List Tag Test',
      'status' => TRUE,
    ]);
    $profile->save();

    // Config entity list tag should exist for entity type.
    $list_cache_tags = $profile->getEntityType()->getListCacheTags();

    // List cache tags for config entities include the config prefix list tag.
    $this->assertNotEmpty($list_cache_tags, 'Entity type has list cache tags.');
  }

  /**
   * Test cacheability metadata can be applied.
   */
  public function testCacheabilityMetadataCanBeApplied(): void {
    $profile = LightgalleryProfile::load('default');

    $cacheability = new CacheableMetadata();
    $cacheability->addCacheableDependency($profile);

    $this->assertContains(
      'config:lightgallery_formatter.profile.default',
      $cacheability->getCacheTags(),
      'Profile cache tags added to metadata.'
    );
  }

  /**
   * Test profile config cache is invalidated on delete.
   */
  public function testProfileCacheInvalidatedOnDelete(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'delete_cache_test',
      'label' => 'Delete Cache Test',
      'status' => TRUE,
    ]);
    $profile->save();

    // Store the cache tag.
    $cache_tag = 'config:lightgallery_formatter.profile.delete_cache_test';

    // Delete the profile.
    $profile->delete();

    // Verify profile no longer exists.
    $this->assertNull(LightgalleryProfile::load('delete_cache_test'), 'Profile deleted.');

    // The cache tag would have been invalidated during delete.
    // We verify the tag format was correct.
    $this->assertStringContainsString('delete_cache_test', $cache_tag, 'Cache tag contained profile ID.');
  }

  /**
   * Test cache contexts are appropriate for config entities.
   */
  public function testCacheContextsAppropriate(): void {
    $profile = LightgalleryProfile::load('default');

    $cache_contexts = $profile->getCacheContexts();

    // Config entities typically don't have user-specific cache contexts.
    // They may be empty or contain general contexts.
    $this->assertIsArray($cache_contexts, 'Cache contexts is an array.');

    // Verify no user-specific contexts (config is same for all users).
    $this->assertNotContains('user', $cache_contexts, 'No user context (config is global).');
  }

  /**
   * Test cache max-age is appropriate.
   */
  public function testCacheMaxAgeAppropriate(): void {
    $profile = LightgalleryProfile::load('default');

    $max_age = $profile->getCacheMaxAge();

    // Config entities typically have permanent cache (until invalidated).
    $this->assertEquals(
      Cache::PERMANENT,
      $max_age,
      'Profile has permanent cache max-age.'
    );
  }

  /**
   * Test multiple profiles have unique cache tags.
   */
  public function testMultipleProfilesHaveUniqueCacheTags(): void {
    $profile1 = LightgalleryProfile::create([
      'id' => 'profile_one',
      'label' => 'Profile One',
      'status' => TRUE,
    ]);
    $profile1->save();

    $profile2 = LightgalleryProfile::create([
      'id' => 'profile_two',
      'label' => 'Profile Two',
      'status' => TRUE,
    ]);
    $profile2->save();

    $tags1 = $profile1->getCacheTags();
    $tags2 = $profile2->getCacheTags();

    $this->assertNotEquals($tags1, $tags2, 'Different profiles have different cache tags.');
    $this->assertContains('config:lightgallery_formatter.profile.profile_one', $tags1, 'Profile 1 has correct tag.');
    $this->assertContains('config:lightgallery_formatter.profile.profile_two', $tags2, 'Profile 2 has correct tag.');
  }

}
