<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;

/**
 * Tests config export/import roundtrip for LightGallery profiles.
 *
 * @group lightgallery_formatter
 */
class LightgalleryConfigRoundtripTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
  }

  /**
   * Test create profile with full plugin config.
   */
  public function testCreateProfileWithFullPluginConfig(): void {
    $full_config = [
      'general' => [
        'loop' => FALSE,
        'preload' => 3,
        'speed' => 600,
        'mode' => 'lg-fade',
        'closable' => TRUE,
        'closeOnTap' => FALSE,
        'escKey' => TRUE,
        'keyPress' => FALSE,
        'showCloseIcon' => TRUE,
        'showMaximizeIcon' => TRUE,
        'hideBarsDelay' => 5000,
        'counter' => FALSE,
        'download' => FALSE,
        'enableDrag' => TRUE,
        'enableSwipe' => TRUE,
        'mousewheel' => TRUE,
      ],
      'thumbnail' => [
        'enabled' => TRUE,
        'animateThumb' => FALSE,
        'image_style' => 'large',
        'thumbWidth' => 120,
        'thumbHeight' => '100px',
        'thumbMargin' => 15,
        'currentPagerPosition' => 'left',
        'alignThumbnails' => 'right',
        'toggleThumb' => TRUE,
        'appendThumbnailsTo' => '.lg-outer',
      ],
    ];

    $profile = LightgalleryProfile::create([
      'id' => 'roundtrip_test',
      'label' => 'Roundtrip Test Profile',
      'status' => TRUE,
      'plugin_settings' => $full_config,
    ]);
    $profile->save();

    // Verify creation.
    $loaded = LightgalleryProfile::load('roundtrip_test');
    $this->assertNotNull($loaded, 'Profile created successfully.');
    $this->assertEquals($full_config, $loaded->getAllPluginSettings(), 'Plugin settings match.');
  }

  /**
   * Test export via config API.
   */
  public function testExportViaConfigApi(): void {
    $full_config = [
      'general' => ['loop' => FALSE, 'speed' => 800],
      'thumbnail' => ['enabled' => TRUE, 'thumbMargin' => 20],
    ];

    $profile = LightgalleryProfile::create([
      'id' => 'export_test',
      'label' => 'Export Test',
      'status' => TRUE,
      'plugin_settings' => $full_config,
    ]);
    $profile->save();

    // Get config via config factory (simulates export).
    $config = $this->config('lightgallery_formatter.profile.export_test');
    $exported = $config->get();

    // Verify exported data structure.
    $this->assertEquals('export_test', $exported['id'], 'ID exported correctly.');
    $this->assertEquals('Export Test', $exported['label'], 'Label exported correctly.');
    $this->assertTrue($exported['status'], 'Status exported correctly.');
    $this->assertArrayHasKey('plugin_settings', $exported, 'plugin_settings exported.');
    $this->assertEquals($full_config, $exported['plugin_settings'], 'plugin_settings values match.');
  }

  /**
   * Test delete profile.
   */
  public function testDeleteProfile(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'delete_test',
      'label' => 'Delete Test',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => ['loop' => TRUE],
      ],
    ]);
    $profile->save();

    // Verify exists.
    $this->assertNotNull(LightgalleryProfile::load('delete_test'), 'Profile exists.');

    // Delete.
    $profile->delete();

    // Verify deleted.
    $this->assertNull(LightgalleryProfile::load('delete_test'), 'Profile deleted.');

    // Config should also be gone.
    $config = $this->config('lightgallery_formatter.profile.delete_test');
    $this->assertTrue($config->isNew(), 'Config no longer exists.');
  }

  /**
   * Test import config creates profile.
   */
  public function testImportConfigCreatesProfile(): void {
    // Simulate config import by creating profile from array data.
    $import_data = [
      'id' => 'imported_profile',
      'label' => 'Imported Profile',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => ['loop' => FALSE, 'speed' => 1000],
        'thumbnail' => ['enabled' => TRUE],
      ],
    ];

    $profile = LightgalleryProfile::create($import_data);
    $profile->save();

    // Verify import.
    $loaded = LightgalleryProfile::load('imported_profile');

    $this->assertNotNull($loaded, 'Imported profile created.');
    $this->assertEquals('Imported Profile', $loaded->label(), 'Label imported.');

    $general = $loaded->getPluginSettings('general');
    $this->assertFalse($general['loop'], 'General loop imported.');
    $this->assertEquals(1000, $general['speed'], 'General speed imported.');

    $thumbnail = $loaded->getPluginSettings('thumbnail');
    $this->assertTrue($thumbnail['enabled'], 'Thumbnail enabled imported.');
  }

  /**
   * Test verify restored correctly after roundtrip.
   */
  public function testVerifyRestoredCorrectlyAfterRoundtrip(): void {
    // Create original profile.
    $original_plugin_settings = [
      'general' => [
        'loop' => FALSE,
        'preload' => 4,
        'speed' => 750,
        'mode' => 'lg-zoom-in',
        'closable' => FALSE,
        'closeOnTap' => TRUE,
        'escKey' => FALSE,
        'keyPress' => TRUE,
        'showCloseIcon' => FALSE,
        'showMaximizeIcon' => TRUE,
        'hideBarsDelay' => 2500,
        'counter' => TRUE,
        'download' => TRUE,
        'enableDrag' => FALSE,
        'enableSwipe' => FALSE,
        'mousewheel' => TRUE,
      ],
      'thumbnail' => [
        'enabled' => TRUE,
        'animateThumb' => TRUE,
        'image_style' => 'medium',
        'thumbWidth' => 100,
        'thumbHeight' => '80px',
        'thumbMargin' => 8,
        'currentPagerPosition' => 'middle',
        'alignThumbnails' => 'middle',
        'toggleThumb' => FALSE,
        'appendThumbnailsTo' => '.lg-components',
      ],
    ];

    $profile = LightgalleryProfile::create([
      'id' => 'full_roundtrip',
      'label' => 'Full Roundtrip Test',
      'status' => TRUE,
      'plugin_settings' => $original_plugin_settings,
    ]);
    $profile->save();

    // Export to array.
    $exported = $profile->toArray();

    // Delete original.
    $profile->delete();
    $this->assertNull(LightgalleryProfile::load('full_roundtrip'), 'Original deleted.');

    // Import from exported data.
    $restored = LightgalleryProfile::create([
      'id' => $exported['id'],
      'label' => $exported['label'],
      'status' => $exported['status'],
      'plugin_settings' => $exported['plugin_settings'],
    ]);
    $restored->save();

    // Verify all values match.
    $loaded = LightgalleryProfile::load('full_roundtrip');

    $this->assertEquals('Full Roundtrip Test', $loaded->label(), 'Label restored.');
    $this->assertEquals(
      $original_plugin_settings,
      $loaded->getAllPluginSettings(),
      'All plugin settings restored correctly.'
    );
  }

  /**
   * Test that config hash is consistent.
   */
  public function testConfigHashConsistent(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'hash_test',
      'label' => 'Hash Test',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => ['loop' => TRUE],
      ],
    ]);
    $profile->save();

    // Get config data.
    $config = $this->config('lightgallery_formatter.profile.hash_test');

    // Config should have a uuid.
    $this->assertNotEmpty($config->get('uuid'), 'Config has UUID.');

    // Save again and verify UUID is preserved.
    $loaded = LightgalleryProfile::load('hash_test');
    $original_uuid = $loaded->uuid();
    $loaded->set('label', 'Updated Hash Test');
    $loaded->save();

    $reloaded = LightgalleryProfile::load('hash_test');
    $this->assertEquals($original_uuid, $reloaded->uuid(), 'UUID preserved across updates.');
  }

}
