<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\lightgallery_formatter\LightgalleryPluginBase;
use Drupal\lightgallery_formatter\LightgalleryPluginInterface;

/**
 * Tests plugin instantiation for LightGallery plugins.
 *
 * @group lightgallery_formatter
 */
class LightgalleryPluginInstanceTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * The plugin manager.
   *
   * @var \Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryPluginManager
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
    $this->pluginManager = $this->container->get('plugin.manager.lightgallery');
  }

  /**
   * Test that General plugin can be instantiated.
   */
  public function testGeneralPluginInstantiation(): void {
    $plugin = $this->pluginManager->createInstance('general');

    $this->assertNotNull($plugin, 'General plugin instance created.');
    $this->assertInstanceOf(LightgalleryPluginInterface::class, $plugin, 'General plugin implements interface.');
    $this->assertInstanceOf(LightgalleryPluginBase::class, $plugin, 'General plugin extends base class.');
    $this->assertEquals('general', $plugin->getPluginId(), 'Plugin ID is correct.');
  }

  /**
   * Test that Thumbnail plugin can be instantiated.
   */
  public function testThumbnailPluginInstantiation(): void {
    $plugin = $this->pluginManager->createInstance('thumbnail');

    $this->assertNotNull($plugin, 'Thumbnail plugin instance created.');
    $this->assertInstanceOf(LightgalleryPluginInterface::class, $plugin, 'Thumbnail plugin implements interface.');
    $this->assertInstanceOf(LightgalleryPluginBase::class, $plugin, 'Thumbnail plugin extends base class.');
    $this->assertEquals('thumbnail', $plugin->getPluginId(), 'Plugin ID is correct.');
  }

  /**
   * Test configuration injection works.
   */
  public function testConfigurationInjection(): void {
    $config = [
      'loop' => FALSE,
      'speed' => 600,
      'mode' => 'lg-fade',
    ];

    $plugin = $this->pluginManager->createInstance('general', $config);
    $retrieved_config = $plugin->getConfiguration();

    $this->assertFalse($retrieved_config['loop'], 'Loop configuration injected.');
    $this->assertEquals(600, $retrieved_config['speed'], 'Speed configuration injected.');
    $this->assertEquals('lg-fade', $retrieved_config['mode'], 'Mode configuration injected.');
  }

  /**
   * Test getConfiguration returns merged with defaults.
   */
  public function testGetConfigurationMergesDefaults(): void {
    // Only provide partial configuration.
    $partial_config = ['loop' => FALSE];

    $plugin = $this->pluginManager->createInstance('general', $partial_config);
    $config = $plugin->getConfiguration();

    // Should have the provided value.
    $this->assertFalse($config['loop'], 'Provided configuration preserved.');

    // Should have defaults for other values.
    $this->assertArrayHasKey('speed', $config, 'Default speed is set.');
    $this->assertArrayHasKey('mode', $config, 'Default mode is set.');
    $this->assertArrayHasKey('preload', $config, 'Default preload is set.');
  }

  /**
   * Test setConfiguration works.
   */
  public function testSetConfiguration(): void {
    $plugin = $this->pluginManager->createInstance('general');

    $new_config = [
      'loop' => FALSE,
      'speed' => 1000,
      'counter' => FALSE,
    ];

    $plugin->setConfiguration($new_config);
    $config = $plugin->getConfiguration();

    $this->assertFalse($config['loop'], 'Loop updated via setConfiguration.');
    $this->assertEquals(1000, $config['speed'], 'Speed updated via setConfiguration.');
    $this->assertFalse($config['counter'], 'Counter updated via setConfiguration.');
  }

  /**
   * Test plugin has access to entity type manager.
   */
  public function testPluginHasEntityTypeManager(): void {
    $plugin = $this->pluginManager->createInstance('thumbnail');

    // The plugin should be able to load image styles (tests DI works).
    // This is tested indirectly - if DI failed, plugin creation would throw.
    $this->assertNotNull($plugin, 'Plugin with dependencies created successfully.');
  }

  /**
   * Test getPluginDefinition returns correct data.
   */
  public function testGetPluginDefinition(): void {
    $plugin = $this->pluginManager->createInstance('general');
    $definition = $plugin->getPluginDefinition();

    $this->assertIsArray($definition, 'Plugin definition is an array.');
    $this->assertEquals('general', $definition['id'], 'Definition has correct ID.');
    $this->assertArrayHasKey('label', $definition, 'Definition has label.');
    $this->assertArrayHasKey('description', $definition, 'Definition has description.');
    $this->assertArrayHasKey('weight', $definition, 'Definition has weight.');
  }

  /**
   * Test all plugins can be instantiated without errors.
   */
  public function testAllPluginsInstantiable(): void {
    $definitions = $this->pluginManager->getDefinitions();

    foreach ($definitions as $plugin_id => $definition) {
      $plugin = $this->pluginManager->createInstance($plugin_id);
      $this->assertInstanceOf(
        LightgalleryPluginInterface::class,
        $plugin,
        "Plugin '{$plugin_id}' instantiated successfully."
      );
    }
  }

}
