<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;

/**
 * Tests plugin settings storage in LightGallery profiles.
 *
 * @group lightgallery_formatter
 */
class LightgalleryProfilePluginStorageTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
  }

  /**
   * Test setPluginSettings persists data.
   */
  public function testSetPluginSettingsPersistsData(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
      'status' => TRUE,
    ]);

    $settings = [
      'loop' => FALSE,
      'speed' => 800,
      'mode' => 'lg-fade',
    ];

    $profile->setPluginSettings('general', $settings);
    $profile->save();

    // Reload the profile.
    $loaded_profile = LightgalleryProfile::load('test_profile');
    $stored_settings = $loaded_profile->getPluginSettings('general');

    $this->assertFalse($stored_settings['loop'], 'Loop setting persisted.');
    $this->assertEquals(800, $stored_settings['speed'], 'Speed setting persisted.');
    $this->assertEquals('lg-fade', $stored_settings['mode'], 'Mode setting persisted.');
  }

  /**
   * Test getPluginSettings retrieves correct data.
   */
  public function testGetPluginSettingsRetrievesCorrectData(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => [
          'loop' => TRUE,
          'speed' => 500,
        ],
        'thumbnail' => [
          'enabled' => TRUE,
          'thumbMargin' => 10,
        ],
      ],
    ]);
    $profile->save();

    // Retrieve settings for each plugin separately.
    $general = $profile->getPluginSettings('general');
    $thumbnail = $profile->getPluginSettings('thumbnail');

    $this->assertTrue($general['loop'], 'General loop retrieved correctly.');
    $this->assertEquals(500, $general['speed'], 'General speed retrieved correctly.');

    $this->assertTrue($thumbnail['enabled'], 'Thumbnail enabled retrieved correctly.');
    $this->assertEquals(10, $thumbnail['thumbMargin'], 'Thumbnail margin retrieved correctly.');
  }

  /**
   * Test getAllPluginSettings returns all plugins.
   */
  public function testGetAllPluginSettingsReturnsAll(): void {
    $all_settings = [
      'general' => [
        'loop' => FALSE,
        'speed' => 600,
      ],
      'thumbnail' => [
        'enabled' => TRUE,
        'animateThumb' => FALSE,
      ],
    ];

    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
      'status' => TRUE,
      'plugin_settings' => $all_settings,
    ]);
    $profile->save();

    $retrieved = $profile->getAllPluginSettings();

    $this->assertIsArray($retrieved, 'getAllPluginSettings returns array.');
    $this->assertArrayHasKey('general', $retrieved, 'General plugin settings present.');
    $this->assertArrayHasKey('thumbnail', $retrieved, 'Thumbnail plugin settings present.');
    $this->assertEquals($all_settings, $retrieved, 'All settings match.');
  }

  /**
   * Test unknown plugin IDs return empty array without breaking profile.
   */
  public function testUnknownPluginIdReturnsEmptyArray(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => ['loop' => TRUE],
      ],
    ]);
    $profile->save();

    // Request settings for non-existent plugin.
    $unknown = $profile->getPluginSettings('nonexistent_plugin');

    $this->assertIsArray($unknown, 'Unknown plugin returns array.');
    $this->assertEmpty($unknown, 'Unknown plugin returns empty array.');

    // Profile should still work for known plugins.
    $general = $profile->getPluginSettings('general');
    $this->assertTrue($general['loop'], 'Known plugin settings still accessible.');
  }

  /**
   * Test setting plugin settings multiple times overwrites previous.
   */
  public function testSettingOverwritesPrevious(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
      'status' => TRUE,
    ]);

    // Set initial settings.
    $profile->setPluginSettings('general', ['loop' => TRUE, 'speed' => 400]);
    $profile->save();

    // Overwrite with new settings.
    $profile->setPluginSettings('general', ['loop' => FALSE, 'speed' => 800]);
    $profile->save();

    $loaded = LightgalleryProfile::load('test_profile');
    $settings = $loaded->getPluginSettings('general');

    $this->assertFalse($settings['loop'], 'Loop was overwritten.');
    $this->assertEquals(800, $settings['speed'], 'Speed was overwritten.');
  }

  /**
   * Test plugin settings survive entity update.
   */
  public function testPluginSettingsSurviveUpdate(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('general', ['loop' => FALSE, 'speed' => 500]);
    $profile->save();

    // Update label but not plugin settings.
    $profile->set('label', 'Updated Profile');
    $profile->save();

    $loaded = LightgalleryProfile::load('test_profile');

    $this->assertEquals('Updated Profile', $loaded->label(), 'Label updated.');
    $settings = $loaded->getPluginSettings('general');
    $this->assertFalse($settings['loop'], 'Plugin settings preserved after update.');
    $this->assertEquals(500, $settings['speed'], 'Speed setting preserved.');
  }

}
