<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\Core\Config\Schema\SchemaCheckTrait;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;

/**
 * Tests configuration schema for LightGallery profiles.
 *
 * @group lightgallery_formatter
 */
class LightgalleryProfileSchemaTest extends KernelTestBase {

  use SchemaCheckTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * The typed config manager.
   *
   * @var \Drupal\Core\Config\TypedConfigManagerInterface
   */
  protected TypedConfigManagerInterface $typedConfigManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
    $this->typedConfigManager = $this->container->get('config.typed');
  }

  /**
   * Test profile config passes schema validation.
   */
  public function testProfileConfigPassesSchemaValidation(): void {
    // Load the default profile.
    $profile = LightgalleryProfile::load('default');
    $this->assertNotNull($profile, 'Default profile exists.');

    // Get config name.
    $config_name = 'lightgallery_formatter.profile.' . $profile->id();
    $config_data = $this->config($config_name)->get();

    // Check schema compliance.
    $errors = $this->checkConfigSchema($this->typedConfigManager, $config_name, $config_data);

    $this->assertTrue($errors === TRUE || empty($errors), 'Default profile passes schema validation.');
  }

  /**
   * Test profile with plugin_settings passes schema validation.
   */
  public function testProfileWithPluginSettingsPassesSchema(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'schema_test',
      'label' => 'Schema Test Profile',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => [
          'loop' => TRUE,
          'speed' => 400,
          'mode' => 'lg-slide',
          'closable' => TRUE,
          'closeOnTap' => TRUE,
          'escKey' => TRUE,
          'keyPress' => TRUE,
          'showCloseIcon' => TRUE,
          'showMaximizeIcon' => FALSE,
          'hideBarsDelay' => 0,
          'counter' => TRUE,
          'download' => TRUE,
          'enableDrag' => TRUE,
          'enableSwipe' => TRUE,
          'mousewheel' => FALSE,
          'preload' => 2,
        ],
        'thumbnail' => [
          'enabled' => TRUE,
          'animateThumb' => TRUE,
          'image_style' => 'thumbnail',
          'thumbWidth' => 100,
          'thumbHeight' => '80px',
          'thumbMargin' => 5,
          'currentPagerPosition' => 'middle',
          'alignThumbnails' => 'middle',
          'toggleThumb' => FALSE,
          'appendThumbnailsTo' => '.lg-components',
        ],
      ],
    ]);
    $profile->save();

    $config_name = 'lightgallery_formatter.profile.schema_test';
    $config_data = $this->config($config_name)->get();

    $errors = $this->checkConfigSchema($this->typedConfigManager, $config_name, $config_data);

    $this->assertTrue($errors === TRUE || empty($errors), 'Profile with plugin settings passes schema validation.');
  }

  /**
   * Test plugin_settings structure is valid.
   */
  public function testPluginSettingsStructureValid(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'structure_test',
      'label' => 'Structure Test',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => ['loop' => FALSE],
        'thumbnail' => ['enabled' => TRUE],
      ],
    ]);
    $profile->save();

    $config_name = 'lightgallery_formatter.profile.structure_test';
    $config = $this->config($config_name);

    // Verify plugin_settings is stored as expected.
    $plugin_settings = $config->get('plugin_settings');

    $this->assertIsArray($plugin_settings, 'plugin_settings is an array.');
    $this->assertArrayHasKey('general', $plugin_settings, 'general plugin settings exist.');
    $this->assertArrayHasKey('thumbnail', $plugin_settings, 'thumbnail plugin settings exist.');
    $this->assertIsArray($plugin_settings['general'], 'general settings is an array.');
    $this->assertIsArray($plugin_settings['thumbnail'], 'thumbnail settings is an array.');
  }

  /**
   * Test no untyped fields warning in schema.
   */
  public function testNoUntypedFieldsWarning(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'untyped_test',
      'label' => 'Untyped Fields Test',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => [
          'loop' => TRUE,
          'speed' => 500,
        ],
      ],
    ]);
    $profile->save();

    $config_name = 'lightgallery_formatter.profile.untyped_test';
    $config_data = $this->config($config_name)->get();

    // Check that the schema exists for the config.
    $has_schema = $this->typedConfigManager->hasConfigSchema($config_name);
    $this->assertTrue($has_schema, 'Schema exists for profile config.');

    // The checkConfigSchema will catch untyped fields.
    $errors = $this->checkConfigSchema($this->typedConfigManager, $config_name, $config_data);

    if ($errors !== TRUE && !empty($errors)) {
      // If there are errors, they should not be about missing schema.
      foreach ($errors as $error) {
        $this->assertStringNotContainsString(
          'missing schema',
          strtolower($error),
          'No missing schema errors.'
        );
      }
    }
  }

  /**
   * Test empty plugin_settings is valid.
   */
  public function testEmptyPluginSettingsValid(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'empty_plugins',
      'label' => 'Empty Plugins Test',
      'status' => TRUE,
      'plugin_settings' => [],
    ]);
    $profile->save();

    $config_name = 'lightgallery_formatter.profile.empty_plugins';
    $config_data = $this->config($config_name)->get();

    $errors = $this->checkConfigSchema($this->typedConfigManager, $config_name, $config_data);
    $this->assertTrue($errors === TRUE || empty($errors), 'Empty plugin_settings passes schema validation.');
  }

}
