<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;

/**
 * Tests security of LightGallery plugin settings.
 *
 * @group lightgallery_formatter
 */
class LightgallerySecurityTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * The plugin manager.
   *
   * @var \Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryPluginManager
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
    $this->pluginManager = $this->container->get('plugin.manager.lightgallery');
  }

  /**
   * Test that HTML in appendThumbnailsTo is stored and passed to JS.
   */
  public function testHtmlInAppendThumbnailsToFieldHandled(): void {
    // Test with a dangerous-looking but valid selector option.
    $test_input = '.lg-outer';

    $profile = LightgalleryProfile::create([
      'id' => 'xss_test',
      'label' => 'XSS Test',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('thumbnail', [
      'enabled' => TRUE,
      'appendThumbnailsTo' => $test_input,
    ]);
    $profile->save();

    $loaded = LightgalleryProfile::load('xss_test');
    $settings = $loaded->getPluginSettings('thumbnail');

    // The value is stored as-is (config storage).
    $this->assertEquals($test_input, $settings['appendThumbnailsTo'], 'Value stored.');

    // When passed to JS settings, it should be included.
    $plugin = $this->pluginManager->createInstance('thumbnail', $settings);
    $js_settings = $plugin->buildJsSettings($settings);

    $this->assertArrayHasKey('appendThumbnailsTo', $js_settings, 'Key present.');
    $this->assertEquals($test_input, $js_settings['appendThumbnailsTo'], 'Value preserved.');
  }

  /**
   * Test script injection in mode setting prevented.
   */
  public function testScriptInjectionInModeSettingNotExecutable(): void {
    $dangerous_mode = '"><script>alert(1)</script><div class="';

    $profile = LightgalleryProfile::create([
      'id' => 'mode_xss',
      'label' => 'Mode XSS Test',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('general', [
      'mode' => $dangerous_mode,
    ]);
    $profile->save();

    $loaded = LightgalleryProfile::load('mode_xss');
    $settings = $loaded->getPluginSettings('general');

    // Value stored as-is in config.
    $this->assertEquals($dangerous_mode, $settings['mode'], 'Mode stored as-is.');

    // In real usage, the mode is used as a CSS class name by LightGallery.
    // Invalid values would just be ignored by the library.
    // The important thing is that it doesn't get rendered as HTML.
    $plugin = $this->pluginManager->createInstance('general', $settings);
    $js_settings = $plugin->buildJsSettings($settings);

    // JS settings contain the value; escaping happens in JS layer.
    $this->assertEquals($dangerous_mode, $js_settings['mode'], 'Mode passed to JS.');
  }

  /**
   * Test that boolean settings are properly typed.
   */
  public function testBooleanSettingsProperlyTyped(): void {
    $plugin = $this->pluginManager->createInstance('general', [
      'loop' => 'true',
      'counter' => '1',
      'download' => 'yes',
    ]);

    $js_settings = $plugin->buildJsSettings($plugin->getConfiguration());

    // All boolean settings should be actual booleans in JS settings.
    $this->assertIsBool($js_settings['loop'], 'Loop is boolean.');
    $this->assertIsBool($js_settings['counter'], 'Counter is boolean.');
    $this->assertIsBool($js_settings['download'], 'Download is boolean.');
  }

  /**
   * Test that integer settings are properly typed.
   */
  public function testIntegerSettingsProperlyTyped(): void {
    $plugin = $this->pluginManager->createInstance('general', [
      'speed' => '500',
      'preload' => '3.5',
      'hideBarsDelay' => 'not_a_number',
    ]);

    $js_settings = $plugin->buildJsSettings($plugin->getConfiguration());

    // Integer settings should be integers in JS settings.
    $this->assertIsInt($js_settings['speed'], 'Speed is integer.');
    $this->assertIsInt($js_settings['preload'], 'Preload is integer.');
    $this->assertIsInt($js_settings['hideBarsDelay'], 'HideBarsDelay is integer.');
  }

  /**
   * Test profile label is properly escaped when displayed.
   */
  public function testProfileLabelContainingHtml(): void {
    $dangerous_label = '<script>alert("xss")</script>Test';

    $profile = LightgalleryProfile::create([
      'id' => 'label_xss',
      'label' => $dangerous_label,
      'status' => TRUE,
    ]);
    $profile->save();

    $loaded = LightgalleryProfile::load('label_xss');

    // The label is stored as-is.
    $this->assertEquals($dangerous_label, $loaded->label(), 'Label stored as-is.');

    // Drupal's rendering system will escape the label when displayed.
    // This test verifies storage; rendering escaping is handled by templates.
  }

  /**
   * Test plugin settings don't allow PHP code execution.
   */
  public function testPluginSettingsDontAllowPhpExecution(): void {
    $php_code = '<?php system("whoami"); ?>';

    $profile = LightgalleryProfile::create([
      'id' => 'php_test',
      'label' => 'PHP Test',
      'status' => TRUE,
    ]);
    // Test with thumbHeight which accepts arbitrary strings.
    $profile->setPluginSettings('thumbnail', [
      'enabled' => TRUE,
      'thumbHeight' => $php_code,
    ]);
    $profile->save();

    $loaded = LightgalleryProfile::load('php_test');
    $settings = $loaded->getPluginSettings('thumbnail');

    // PHP code is stored as string, not executed.
    $this->assertEquals($php_code, $settings['thumbHeight'], 'PHP code stored as string.');

    // Creating plugin and building JS settings should not execute PHP.
    $plugin = $this->pluginManager->createInstance('thumbnail', $settings);
    $js_settings = $plugin->buildJsSettings($settings);

    // Still just a string.
    $this->assertArrayHasKey('thumbHeight', $js_settings, 'ThumbHeight key exists.');
    $this->assertEquals($php_code, $js_settings['thumbHeight'], 'PHP code not executed.');
  }

  /**
   * Test that extreme numeric values are handled safely.
   */
  public function testExtremeNumericValuesHandled(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'extreme_test',
      'label' => 'Extreme Test',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('general', [
      'speed' => PHP_INT_MAX,
      'preload' => -PHP_INT_MAX,
      'hideBarsDelay' => 999999999999,
    ]);
    $profile->save();

    $loaded = LightgalleryProfile::load('extreme_test');
    $plugin = $this->pluginManager->createInstance('general', $loaded->getPluginSettings('general'));
    $js_settings = $plugin->buildJsSettings($loaded->getPluginSettings('general'));

    // Values should be integers, even if extreme.
    $this->assertIsInt($js_settings['speed'], 'Extreme speed is integer.');
    $this->assertIsInt($js_settings['preload'], 'Extreme preload is integer.');
    $this->assertIsInt($js_settings['hideBarsDelay'], 'Extreme hideBarsDelay is integer.');
  }

  /**
   * Test that null values are handled gracefully.
   */
  public function testNullValuesHandledGracefully(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'null_test',
      'label' => 'Null Test',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('general', [
      'loop' => NULL,
      'speed' => NULL,
      'mode' => NULL,
    ]);
    $profile->save();

    $loaded = LightgalleryProfile::load('null_test');
    $plugin = $this->pluginManager->createInstance('general', $loaded->getPluginSettings('general'));
    $js_settings = $plugin->buildJsSettings($loaded->getPluginSettings('general'));

    // Null values should fall back to defaults or be coerced.
    // The plugin should not crash.
    $this->assertNotNull($js_settings, 'JS settings generated despite null inputs.');
    $this->assertArrayHasKey('loop', $js_settings, 'Loop key exists.');
    $this->assertArrayHasKey('speed', $js_settings, 'Speed key exists.');
  }

}
