<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\KernelTests\KernelTestBase;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;

/**
 * Tests graceful handling of third-party plugin failures.
 *
 * @group lightgallery_formatter
 */
class LightgalleryThirdPartyPluginFailureTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * The plugin manager.
   *
   * @var \Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryPluginManager
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
    $this->pluginManager = $this->container->get('plugin.manager.lightgallery');
  }

  /**
   * Test requesting non-existent plugin throws exception.
   */
  public function testNonExistentPluginThrowsException(): void {
    $this->expectException(PluginNotFoundException::class);
    $this->pluginManager->createInstance('nonexistent_plugin');
  }

  /**
   * Test profile with missing plugin ID loads correctly.
   */
  public function testProfileWithMissingPluginLoads(): void {
    // Create profile with settings for a plugin that doesn't exist.
    $profile = LightgalleryProfile::create([
      'id' => 'missing_plugin_test',
      'label' => 'Missing Plugin Test',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => ['loop' => FALSE],
        'removed_plugin' => ['some_setting' => TRUE],
      ],
    ]);
    $profile->save();

    // Profile should load without error.
    $loaded = LightgalleryProfile::load('missing_plugin_test');

    $this->assertNotNull($loaded, 'Profile with missing plugin loads.');
    $this->assertEquals('Missing Plugin Test', $loaded->label(), 'Label correct.');
  }

  /**
   * Test iteration skips missing plugins gracefully.
   */
  public function testIterationSkipsMissingPlugins(): void {
    // Create profile with both existing and non-existing plugin settings.
    $profile = LightgalleryProfile::create([
      'id' => 'skip_test',
      'label' => 'Skip Test',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => ['loop' => TRUE, 'speed' => 500],
        'missing_plugin' => ['enabled' => TRUE],
        'thumbnail' => ['enabled' => FALSE],
      ],
    ]);
    $profile->save();

    // Iterate only through discovered plugins.
    $js_settings = [];
    $errors = [];

    foreach ($this->pluginManager->getSortedDefinitions() as $plugin_id => $definition) {
      try {
        $plugin_settings = $profile->getPluginSettings($plugin_id);
        $plugin = $this->pluginManager->createInstance($plugin_id, $plugin_settings);

        if ($plugin->isEnabled($plugin_settings)) {
          $js_settings = array_merge($js_settings, $plugin->buildJsSettings($plugin_settings));
        }
      }
      catch (\Exception $e) {
        $errors[] = $e->getMessage();
      }
    }

    // Should have general settings but not error about missing plugin.
    $this->assertArrayHasKey('loop', $js_settings, 'General settings collected.');
    $this->assertEmpty($errors, 'No errors during iteration.');
  }

  /**
   * Test other plugins unaffected by missing plugin data.
   */
  public function testOtherPluginsUnaffected(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'unaffected_test',
      'label' => 'Unaffected Test',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => ['loop' => FALSE, 'speed' => 800],
        'orphaned_plugin_data' => ['value' => 'should_be_ignored'],
      ],
    ]);
    $profile->save();

    // General plugin should work normally.
    $plugin = $this->pluginManager->createInstance('general', $profile->getPluginSettings('general'));
    $js_settings = $plugin->buildJsSettings($profile->getPluginSettings('general'));

    $this->assertFalse($js_settings['loop'], 'General loop setting correct.');
    $this->assertEquals(800, $js_settings['speed'], 'General speed setting correct.');
  }

  /**
   * Test getPluginSettings returns empty for missing plugin.
   */
  public function testGetPluginSettingsReturnsEmptyForMissing(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'empty_test',
      'label' => 'Empty Test',
      'status' => TRUE,
    ]);
    $profile->save();

    // Request settings for non-existent plugin.
    $settings = $profile->getPluginSettings('never_existed');

    $this->assertIsArray($settings, 'Returns array for missing plugin.');
    $this->assertEmpty($settings, 'Returns empty array for missing plugin.');
  }

  /**
   * Test safe iteration pattern for formatter.
   */
  public function testSafeIterationPatternForFormatter(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'safe_iteration',
      'label' => 'Safe Iteration',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('general', ['loop' => TRUE]);
    $profile->setPluginSettings('thumbnail', ['enabled' => TRUE]);
    $profile->save();

    // This is the recommended pattern for formatter to use.
    // Note: With React, all libs are bundled; plugins return empty arrays.
    $libraries = ['lightgallery_formatter/gallery'];
    $gallery_settings = [];

    foreach ($this->pluginManager->getSortedDefinitions() as $plugin_id => $definition) {
      $plugin_settings = $profile->getPluginSettings($plugin_id);
      $plugin = $this->pluginManager->createInstance($plugin_id, $plugin_settings);

      if ($plugin->isEnabled($plugin_settings)) {
        $gallery_settings = array_merge(
          $gallery_settings,
          $plugin->buildJsSettings($plugin_settings)
        );
      }
    }

    // Verify the pattern works correctly.
    $this->assertNotEmpty($gallery_settings, 'Gallery settings collected.');
    $this->assertArrayHasKey('loop', $gallery_settings, 'General loop present.');
    $this->assertArrayHasKey('thumbnail', $gallery_settings, 'Thumbnail present.');
    // Main gallery library is always attached by the formatter.
    $this->assertContains('lightgallery_formatter/gallery', $libraries, 'Gallery library present.');
  }

  /**
   * Test profile deletion with plugin settings.
   */
  public function testProfileDeletionWithPluginSettings(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'delete_with_plugins',
      'label' => 'Delete With Plugins',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => ['loop' => TRUE],
        'thumbnail' => ['enabled' => TRUE],
        'future_plugin' => ['data' => 'value'],
      ],
    ]);
    $profile->save();

    // Delete should work without issues.
    $profile->delete();

    $this->assertNull(
      LightgalleryProfile::load('delete_with_plugins'),
      'Profile with plugin settings deleted successfully.'
    );
  }

  /**
   * Test plugin manager handles empty definitions.
   */
  public function testPluginManagerHandlesEmptyDefinitions(): void {
    $definitions = $this->pluginManager->getDefinitions();

    // Should have at least the built-in plugins.
    $this->assertNotEmpty($definitions, 'Plugin manager returns definitions.');
    $this->assertGreaterThanOrEqual(2, count($definitions), 'At least 2 built-in plugins.');
  }

  /**
   * Test createInstances handles mixed plugin availability.
   */
  public function testCreateInstancesHandlesMixedAvailability(): void {
    // createInstances only creates instances for discovered plugins.
    $instances = $this->pluginManager->createInstances();

    // Should have instances for all discovered plugins.
    $this->assertArrayHasKey('general', $instances, 'General plugin instantiated.');
    $this->assertArrayHasKey('thumbnail', $instances, 'Thumbnail plugin instantiated.');

    // Should NOT have instances for non-discovered plugins.
    $this->assertArrayNotHasKey('nonexistent', $instances, 'Non-existent plugin not instantiated.');
  }

}
