# LightGallery Formatter

A Drupal module that provides a field formatter to display media image fields using the [LightGallery](https://www.lightgalleryjs.com/) JavaScript library.

## Features

- **Field Formatter for Media Fields**: Display entity reference fields (targeting media image bundles) as beautiful lightbox galleries.
- **React-Powered**: Modern React-based rendering with all dependencies bundled - no external library installation required.
- **Progressive Enhancement**: Static images render immediately as fallback while React loads, ensuring SEO and accessibility.
- **Profile-based Configuration**: Create multiple reusable gallery profiles with different settings.
- **Plugin Architecture**: Modular system for LightGallery features with 11 built-in plugins.
- **Two Thumbnail Styles**: Separate image styles for gallery grid (inline) and lightbox thumbnail strip.
- **Image Style Support**: Apply Drupal image styles to gallery thumbnails.
- **Responsive Design**: Works seamlessly on desktop and mobile devices.
- **Remote Video Support**: Automatically extracts thumbnails from YouTube and Vimeo videos.

## Requirements

- Drupal 10.3+ or Drupal 11
- Media module (core)
- Node.js 22+ (for development/building only)

**Note:** LightGallery and React are bundled with the module - no external library installation required.

## Installation

### Using Composer (Recommended)

```bash
composer require drupal/lightgallery_formatter
drush en lightgallery_formatter -y
```

### Manual Installation

1. Download the module from [drupal.org/project/lightgallery_formatter](https://www.drupal.org/project/lightgallery_formatter)
2. Extract to `modules/contrib/lightgallery_formatter`
3. Enable via Admin UI or Drush: `drush en lightgallery_formatter -y`

That's it! The module includes all required JavaScript libraries pre-bundled.

## Configuration

### Step 1: Create a Gallery Profile

1. Navigate to **Structure → LightGallery Profiles** (`/admin/structure/lightgallery-profiles`)
2. Click **Add profile**
3. Configure **General Settings**:
   - **License Key**: Your LightGallery license key (use `0000-0000-000-0000` for non-commercial)
   - **Loop**: Enable infinite looping through images
   - **Speed**: Animation speed in milliseconds
   - **Mode**: Slide transition mode (30+ options)
   - **Allow Media Overlap**: Keep enabled to see plugin icons in the toolbar
   - **Controls**: Show/hide close, maximize, download buttons
4. Configure plugins (optional):
   - **Thumbnail Navigation**: Thumbnail strip at bottom of lightbox
   - **Zoom**: Zoom in/out functionality
   - **Fullscreen**: Full browser window display
   - **Pager**: Page number indicator
   - **Rotate**: Image rotation controls
   - **Autoplay**: Automatic slideshow
   - **Share**: Social sharing buttons
   - **Hash**: URL hash navigation for deep linking
   - **Relative Caption**: Position-relative captions
   - **Medium Zoom**: Medium.com-style zoom effect
5. Save the profile

### Step 2: Configure Your Field Display

1. Go to **Structure → Content types → [Your Type] → Manage display**
2. Find your media image field
3. Set the formatter to **LightGallery**
4. Click the gear icon to configure:
   - Select a **LightGallery Profile**
   - Choose **Gallery thumbnails (inline on page)** image style - for the grid displayed on the page
   - Configure **Gallery item source** for each media type (optional)
5. Save

**Note:** There are two separate thumbnail settings:
- **Gallery thumbnails** (Field Formatter): Image style for the inline grid on the page
- **Lightbox thumbnails** (Profile → Thumbnail Navigation): Image style for the thumbnail strip at the bottom of the lightbox

### Step 3: View Your Content

Create or view content with media images — they will now display as a LightGallery!

## Available Plugins

The module includes 11 plugins for configuring LightGallery features:

| Plugin | Description | Weight |
|--------|-------------|--------|
| **General** | Core settings (loop, speed, transitions, controls) | -100 |
| **Thumbnail** | Thumbnail strip navigation at bottom of lightbox | 0 |
| **Zoom** | Zoom in/out, actual size, infinite zoom | 10 |
| **Fullscreen** | Full browser window display | 20 |
| **Pager** | Page number/count indicator | 25 |
| **Relative Caption** | Captions positioned relative to image | 30 |
| **Rotate** | Image rotation and flip controls | 35 |
| **Autoplay** | Automatic slideshow with progress bar | 40 |
| **Medium Zoom** | Medium.com-style zoom experience | 45 |
| **Share** | Social sharing (Facebook, Twitter, Pinterest) | 50 |
| **Hash** | URL hash navigation for deep linking | 55 |

### Hash Plugin: Auto-Generated Gallery ID

The Hash plugin enables URL-based navigation, allowing users to share direct links to specific images (e.g., `example.com/gallery#lg=gallery-123&slide=2`).

**Why is the Gallery ID auto-generated?**

Unlike the original LightGallery library where you manually set a `galleryId`, this module automatically generates a unique ID for each gallery based on:
- The field name (e.g., `field_images`)
- The entity context

**Benefits of auto-generation:**
- **Prevents conflicts**: Multiple galleries on the same page each get unique IDs
- **Consistency**: The URL hash always matches the actual gallery container element
- **No configuration needed**: Works correctly without manual setup

**Example URL format:**
```
https://example.com/node/123#lg=lightgallery-abc123-field-images&slide=0
```

The format is `#lg={gallery-id}&slide={index}` where:
- `gallery-id` is the auto-generated unique identifier
- `slide` is the zero-based image index

### Known Plugin Interactions

Some plugins may interact unexpectedly when used together:

- **Hash + Thumbnail Navigation**: The hash plugin updates the URL when navigating, which may cause unexpected behavior in some browsers.
- **Relative Caption**: May show console warnings (`getBoundingClientRect`) during slide transitions - this doesn't affect functionality.
- **Medium Zoom + Thumbnails**: The medium zoom click behavior may conflict with thumbnail strip clicks.

If you experience issues, try disabling conflicting plugins in your profile settings.

## Submodules

### LightGallery Formatter Demo

Provides demo content for testing the module with the `demo_umami` installation profile.

```bash
drush en lightgallery_formatter_demo -y
```

### LightGallery Formatter Preview

Provides a live preview feature when configuring gallery profiles.

```bash
drush en lightgallery_formatter_preview -y
```

## Similar Projects

- **[Colorbox](https://www.drupal.org/project/colorbox)** — Classic lightbox solution for Drupal
- **[PhotoSwipe](https://www.drupal.org/project/photoswipe)** — JavaScript gallery with gestures support
- **[GLightbox](https://www.drupal.org/project/glightbox)** — Modern lightbox with video support

## What Makes LightGallery Different?

- **React-Powered**: Modern React-based rendering with all dependencies bundled
- **Progressive Enhancement**: Static images visible immediately, enhanced when JS loads
- **Zero Configuration**: Works out of the box - no external library installation needed
- **Modern JavaScript**: Built with ES modules and Vite, no jQuery dependency
- **Rich Feature Set**: Zoom, fullscreen, thumbnails, autoplay, share buttons, rotate, hash navigation
- **Video Support**: YouTube, Vimeo, and HTML5 video
- **Touch Gestures**: Swipe, pinch-to-zoom on mobile
- **Drupal Plugin System**: Modular Drupal plugins for configuring LightGallery features
- **30+ Transition Modes**: Slide, fade, zoom, rotate, and many more effects
- **SEO Friendly**: Static fallback ensures search engines can index images

## Support

- **Issue queue**: [drupal.org/project/issues/lightgallery_formatter](https://www.drupal.org/project/issues/lightgallery_formatter)
- **Documentation**: [drupal.org/project/lightgallery_formatter](https://www.drupal.org/project/lightgallery_formatter)

## Credits

- **LightGallery Library**: [lightgalleryjs.com](https://www.lightgalleryjs.com/) by Sachin Neravath
- **Module Maintainer**: [man-1982](https://www.drupal.org/u/man-1982)

---

## For Developers

### Setting up a Local Development Environment using DDEV

The project uses [ddev-drupal-contrib](https://github.com/ddev/ddev-drupal-contrib) addon to deploy and debug code.

**Prerequisites:**

* [Docker](https://docs.docker.com/get-docker/) installed.
* [DDEV](https://ddev.readthedocs.io/en/latest/users/install/ddev-installation/) installed.
* [Node.js 22+](https://nodejs.org/) (see `.nvmrc` for exact version)

### Quick Start (Recommended)

The fastest way to get up and running with a fully configured demo site:

```bash
# Clone the repository
git clone git@git.drupal.org:project/lightgallery_formatter.git
cd lightgallery_formatter

# Use correct Node.js version (if using nvm)
nvm use

# Start DDEV
ddev start

# Initialize everything with one command
ddev lgf:init
```

This command will:
- Install Composer dependencies
- Symlink the module to Drupal
- Install Drupal with the **demo_umami** profile
- Enable the LightGallery Formatter module with demo content
- Enable Admin Toolbar for easier navigation
- Provide login credentials (`admin` / `admin`) and a one-time login link

### Manual Setup (Alternative)

If you prefer step-by-step control or need a custom installation:

1. **Clone and start DDEV:**
   ```bash
   git clone git@git.drupal.org:project/lightgallery_formatter.git
   cd lightgallery_formatter
   ddev start
   ```

2. **Install Composer dependencies:**
   ```bash
   ddev poser
   ```

3. **Install Drupal:**
   ```bash
   # Using standard profile
   ddev drush site:install --yes

   # OR using demo_umami profile for demo content
   ddev drush site:install demo_umami --yes
   ```

4. **Enable the module:**
   ```bash
   ddev drush en lightgallery_formatter -y
   ```

5. **Get a login link:**
   ```bash
   ddev drush user:login
   ```

### Useful Commands

| Command | Description |
|---------|-------------|
| `ddev lgf:init` | Full reset and fresh install with demo site |
| `ddev drush cr` | Clear Drupal caches |
| `ddev drush uli` | Get one-time admin login link |
| `ddev phpcs` | Check coding standards |
| `ddev phpcbf` | Auto-fix coding standard violations |
| `ddev phpstan` | Run static analysis |
| `ddev phpunit` | Run PHPUnit tests |
| `ddev describe` | Show project info and URLs |

### Working with composer.contrib.json

After running `ddev lgf:init`, the `composer.contrib.json` file is preserved for manual modifications. This file contains Drupal core and dev dependencies while keeping the module's `composer.json` clean for drupal.org publication.

**Add additional packages:**
```bash
ddev exec "COMPOSER=composer.contrib.json composer require drupal/devel"
```

**Update dependencies:**
```bash
ddev exec "COMPOSER=composer.contrib.json composer update"
```

**Why two composer files?**
- `composer.json` — Module dependencies only (published to drupal.org)
- `composer.contrib.json` — Development environment (Drupal core + dev tools + module)

### Architecture

The module uses a plugin-based architecture with **React** as the rendering engine:

**Backend (Drupal):**
```
src/
├── LightgalleryPluginManager.php      — Plugin manager (Drupal Core pattern)
├── LightgalleryPluginInterface.php    — Plugin interface
├── LightgalleryPluginBase.php         — Base class for plugins
├── Entity/
│   └── LightgalleryProfile.php        — Config entity for profiles
├── Plugin/
│   ├── Field/FieldFormatter/
│   │   └── LightGalleryFormatter.php  — Field formatter
│   └── Lightgallery/
│       ├── General.php                — General settings plugin
│       ├── Thumbnail.php              — Thumbnail navigation plugin
│       ├── Zoom.php                   — Zoom plugin
│       ├── Fullscreen.php             — Fullscreen plugin
│       ├── Pager.php                  — Pager plugin
│       ├── RelativeCaption.php        — Relative caption plugin
│       ├── Rotate.php                 — Rotate plugin
│       ├── Autoplay.php               — Autoplay plugin
│       ├── MediumZoom.php             — Medium zoom plugin
│       ├── Share.php                  — Share plugin
│       └── Hash.php                   — Hash plugin
└── Form/
    └── LightgalleryProfileForm.php    — Profile configuration form

templates/
└── lightgallery-formatter.html.twig   — Gallery template with fallback
```

**Frontend (React):**
```
js/
├── react/
│   ├── GalleryApp.jsx                 — React LightGallery component
│   └── bootstrap.jsx                  — Drupal behavior for mounting
└── dist/
    ├── lightgallery-react.js          — ES module bundle (React + LightGallery)
    └── style.css                      — Bundled LightGallery CSS
```

**Progressive Enhancement Flow:**
1. Drupal renders static images as `<a>` links wrapping `<img>` tags (SEO/accessibility fallback)
2. Gallery configuration is passed via `drupalSettings.lightgallery_formatter.galleries`
3. React reads the configuration and replaces the static content with interactive gallery
4. Plugin flags control which LightGallery plugins are enabled

**Template System:**
The module uses a Twig template (`lightgallery-formatter.html.twig`) that:
- Renders static image links as immediate fallback
- Attaches the React library via `{{ attach_library() }}`
- Provides theme override points for customization

### Building the React Bundle

The module includes pre-built JavaScript, but if you need to modify the React components:

```bash
# Use correct Node.js version
nvm use  # Reads from .nvmrc (Node.js 22)

# Install Node.js dependencies
npm install

# Build the production bundle (JS + CSS)
npm run build

# Watch mode for development
npm run dev

# Build JS only
npm run build:js

# Build CSS only (SCSS compilation)
npm run build:css

# Watch CSS changes
npm run watch:css
```

**Build output:**
- `js/dist/lightgallery-react.js` — Minified ES module (React + LightGallery + all plugins)
- `js/dist/style.css` — LightGallery CSS (bundled from npm)
- `css/lightgallery_formatter.theme.css` — Module theme (compiled from SCSS)

**Note:** The built files in `js/dist/` are committed to the repository so the module works without requiring a build step for end users.

**ES Module Format:**
The JavaScript bundle is built as an ES module (`type: module` in Drupal's libraries.yml). This:
- Enables modern JavaScript features
- Prevents Drupal's JS aggregation from interfering
- Provides better code splitting and tree-shaking

**Why bundle React?**
React and LightGallery are bundled together to ensure version compatibility and eliminate external dependencies. This makes the module work reliably without requiring users to install additional libraries.

### Adding New Plugins

To add a new LightGallery plugin:

1. **Create Drupal plugin** in `src/Plugin/Lightgallery/`:
   ```php
   <?php
   namespace Drupal\lightgallery_formatter\Plugin\Lightgallery;
   
   use Drupal\lightgallery_formatter\LightgalleryPluginBase;
   
   /**
    * @LightgalleryPlugin(
    *   id = "my_plugin",
    *   label = @Translation("My Plugin"),
    *   description = @Translation("Description here."),
    *   weight = 60
    * )
    */
   class MyPlugin extends LightgalleryPluginBase {
     // Implement defaultSettings(), buildForm(), buildJsSettings(), etc.
   }
   ```

2. **Update React component** in `js/react/GalleryApp.jsx`:
   ```jsx
   import lgMyPlugin from 'lightgallery/plugins/myPlugin';
   import 'lightgallery/css/lg-my-plugin.css';  // if CSS exists
   
   const AVAILABLE_PLUGINS = {
     // ... existing plugins
     my_plugin: lgMyPlugin,
   };
   ```

3. **Update schema** in `config/schema/lightgallery_formatter.schema.yml`

4. **Update tests** in `tests/src/Kernel/LightgalleryPluginDiscoveryTest.php`

5. **Rebuild** assets: `npm run build && ddev drush cr`
