/**
 * @file
 * React LightGallery component for Drupal integration.
 */

import { useCallback } from 'react';
import LightGallery from 'lightgallery/react';

// Import LightGallery plugins.
// Only import plugins that have corresponding Drupal plugin implementations.
import lgThumbnail from 'lightgallery/plugins/thumbnail';
import lgZoom from 'lightgallery/plugins/zoom';
import lgFullscreen from 'lightgallery/plugins/fullscreen';
import lgPager from 'lightgallery/plugins/pager';
import lgRelativeCaption from 'lightgallery/plugins/relativeCaption';
import lgRotate from 'lightgallery/plugins/rotate';
import lgAutoplay from 'lightgallery/plugins/autoplay';
import lgMediumZoom from 'lightgallery/plugins/mediumZoom';
import lgShare from 'lightgallery/plugins/share';
import lgHash from 'lightgallery/plugins/hash';

// Future plugins (add when Drupal plugins are implemented):
// import lgVideo from 'lightgallery/plugins/video';

// Import core LightGallery styles.
import 'lightgallery/css/lightgallery.css';
import 'lightgallery/css/lg-thumbnail.css';
import 'lightgallery/css/lg-zoom.css';
import 'lightgallery/css/lg-fullscreen.css';
import 'lightgallery/css/lg-pager.css';
import 'lightgallery/css/lg-relative-caption.css';
import 'lightgallery/css/lg-rotate.css';
import 'lightgallery/css/lg-autoplay.css';
import 'lightgallery/css/lg-medium-zoom.css';
import 'lightgallery/css/lg-share.css';

// Future plugin styles (add when Drupal plugins are implemented):
// import 'lightgallery/css/lg-video.css';

/**
 * Map of available LightGallery plugins.
 *
 * Keys match the plugin IDs from Drupal's plugin system.
 * Add new plugins here when implementing corresponding Drupal plugins.
 */
const AVAILABLE_PLUGINS = {
  thumbnail: lgThumbnail,
  zoom: lgZoom,
  fullscreen: lgFullscreen,
  pager: lgPager,
  relative_caption: lgRelativeCaption,
  rotate: lgRotate,
  autoplay: lgAutoplay,
  medium_zoom: lgMediumZoom,
  share: lgShare,
  hash: lgHash,
  // Future plugins:
  // video: lgVideo,
};

/**
 * Resolves enabled plugins based on plugin flags from Drupal settings.
 *
 * @param {Object} pluginFlags - Object with plugin IDs as keys and boolean enabled state.
 * @returns {Array} Array of LightGallery plugin instances.
 */
function resolvePlugins(pluginFlags = {}) {
  const enabledPlugins = [];

  Object.entries(pluginFlags).forEach(([pluginId, isEnabled]) => {
    if (isEnabled && AVAILABLE_PLUGINS[pluginId]) {
      enabledPlugins.push(AVAILABLE_PLUGINS[pluginId]);
    }
  });

  return enabledPlugins;
}

/**
 * GalleryApp React component.
 *
 * Renders a LightGallery instance with inline thumbnails.
 * Clicking on thumbnails opens the lightbox overlay.
 *
 * @param {Object} props - Component props.
 * @param {Array} props.items - Array of gallery items.
 * @param {Object} props.options - LightGallery configuration options.
 * @param {Object} props.plugins - Plugin flags from Drupal (e.g., { thumbnail: true }).
 * @returns {JSX.Element} The LightGallery component.
 */
export default function GalleryApp({ items = [], options = {}, plugins = {} }) {
  // Resolve which LightGallery plugins to enable.
  const resolvedPlugins = resolvePlugins(plugins);

  /**
   * Handle gallery initialization.
   *
   * @param {Object} detail - Event detail containing LightGallery instance.
   */
  const onInit = useCallback((detail) => {
    // Gallery instance is available at detail.instance if needed.
    if (typeof window !== 'undefined' && window.Drupal && window.Drupal.behaviors) {
      // Dispatch custom event for other Drupal modules to hook into.
      const event = new CustomEvent('lightgallery:init', {
        detail: { instance: detail.instance },
        bubbles: true,
      });
      document.dispatchEvent(event);
    }
  }, []);

  /**
   * Handle click on gallery items.
   * Prevents default anchor behavior to avoid navigation conflicts.
   *
   * @param {Event} e - Click event.
   */
  const handleItemClick = useCallback((e) => {
    // Prevent default anchor navigation - LightGallery handles this.
    e.preventDefault();
  }, []);

  // If no items, don't render anything.
  if (!items || items.length === 0) {
    return null;
  }

  // Build the LightGallery options.
  // License key and other settings come from Drupal's General plugin configuration.
  const gallerySettings = {
    // Plugins to use.
    plugins: resolvedPlugins,
    // Tell the thumbnail plugin to use data-thumb attribute for lightbox thumbnails.
    // This ensures the lightbox thumbnail strip uses the correct image style.
    exThumbImage: 'data-thumb',
    // Prevent gallery from closing when clicking on dark areas around the image.
    // This helps prevent accidental closes especially with hash plugin.
    closeOnTap: false,
    // Spread Drupal-configured options (includes licenseKey, speed, etc.).
    ...options,
  };

  return (
    <LightGallery
      onInit={onInit}
      {...gallerySettings}
    >
      {items.map((item, index) => (
        <a
          key={item.id || index}
          href={item.src}
          data-src={item.src}
          data-sub-html={item.subHtml || item.alt || ''}
          // Lightbox thumbnail - used for thumbnail strip at bottom of lightbox.
          // Falls back to gallery thumbnail if not set.
          data-thumb={item.lightboxThumb || item.thumb || item.src}
          className="lightgallery-item"
          onClick={handleItemClick}
        >
          {/* Gallery thumbnail - displayed inline on the page. */}
          <img
            src={item.thumb || item.src}
            alt={item.alt || ''}
            className="lightgallery-thumbnail"
          />
        </a>
      ))}
    </LightGallery>
  );
}
