/**
 * @file
 * Bootstrap script for mounting React LightGallery components in Drupal.
 *
 * This file reads gallery configurations from drupalSettings and mounts
 * React components on the appropriate container elements.
 */

import { createRoot } from 'react-dom/client';
import GalleryApp from './GalleryApp.jsx';

/**
 * Storage for React roots to prevent double mounting and enable cleanup.
 */
const galleryRoots = new Map();

/**
 * Mount a React gallery component on a container element.
 *
 * This function clears the fallback static content and replaces it
 * with the interactive React gallery. The fallback content provides:
 * - SEO: Search engines can index images and links
 * - Accessibility: Works without JavaScript
 * - Performance: Users see images immediately before JS loads
 *
 * @param {HTMLElement} container - The DOM element to mount the gallery on.
 * @param {string} galleryId - Unique identifier for the gallery.
 * @param {Object} config - Gallery configuration from drupalSettings.
 */
function mountGallery(container, galleryId, config) {
  // Prevent double mounting.
  if (galleryRoots.has(galleryId)) {
    return;
  }

  try {
    // Clear the fallback static content before mounting React.
    // The fallback content (static images with links) was rendered
    // server-side in Twig for SEO and no-JS accessibility.
    container.innerHTML = '';

    // Add class to indicate React has taken over.
    container.classList.add('lightgallery-initialized');

    const root = createRoot(container);
    root.render(
      <GalleryApp
        items={config.items || []}
        options={config.options || {}}
        plugins={config.plugins || {}}
      />
    );
    galleryRoots.set(galleryId, root);
  } catch (error) {
    console.error(`[LightGallery] Failed to mount gallery "${galleryId}":`, error);
    // On error, restore visibility of fallback content.
    container.classList.remove('lightgallery-initialized');
  }
}

/**
 * Unmount a React gallery component.
 *
 * @param {string} galleryId - Unique identifier for the gallery to unmount.
 */
function unmountGallery(galleryId) {
  const root = galleryRoots.get(galleryId);
  if (root) {
    root.unmount();
    galleryRoots.delete(galleryId);
  }
}

/**
 * Process all gallery configurations and mount components.
 *
 * @param {HTMLElement|Document} context - The DOM context to search within.
 */
function processGalleries(context) {
  // Get gallery configurations from drupalSettings.
  const galleries = window.drupalSettings?.lightgallery_formatter?.galleries;

  if (!galleries) {
    return;
  }

  // Find and mount each gallery.
  Object.entries(galleries).forEach(([galleryId, config]) => {
    // Skip if already mounted.
    if (galleryRoots.has(galleryId)) {
      return;
    }

    // Find container by data attribute.
    const container = context.querySelector
      ? context.querySelector(`[data-lightgallery-id="${galleryId}"]`)
      : document.querySelector(`[data-lightgallery-id="${galleryId}"]`);

    if (container) {
      mountGallery(container, galleryId, config);
    }
  });
}

/**
 * Drupal behavior for LightGallery React integration.
 *
 * This behavior is triggered on page load and AJAX updates.
 */
if (window.Drupal && window.Drupal.behaviors) {
  window.Drupal.behaviors.lightgalleryReact = {
    attach(context) {
      // Use once() if available to prevent multiple processing.
      if (window.once) {
        const elements = window.once(
          'lightgallery-react',
          '[data-lightgallery-id]',
          context
        );
        elements.forEach((element) => {
          const galleryId = element.dataset.lightgalleryId;
          const config = window.drupalSettings?.lightgallery_formatter?.galleries?.[galleryId];
          if (config) {
            mountGallery(element, galleryId, config);
          }
        });
      } else {
        // Fallback without once().
        processGalleries(context);
      }
    },

    detach(context, settings, trigger) {
      // Clean up on AJAX removal.
      if (trigger === 'unload') {
        const containers = context.querySelectorAll
          ? context.querySelectorAll('[data-lightgallery-id]')
          : [];

        containers.forEach((container) => {
          const galleryId = container.dataset.lightgalleryId;
          unmountGallery(galleryId);
        });
      }
    },
  };
}

// Export for potential external use.
export { mountGallery, unmountGallery, processGalleries };

