<?php

namespace Drupal\lightgallery_formatter_preview;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Component\Utility\Html;
use Drupal\file\FileInterface;
use Drupal\media\MediaInterface;

/**
 * Service for building LightGallery preview from Media entities.
 *
 * This service builds previews for the profile form using the React-based
 * LightGallery component.
 */
class PreviewBuilder {

  use StringTranslationTrait;

  /**
   * The state key for storing preview media IDs.
   */
  const STATE_KEY = 'lightgallery_formatter_preview.media_ids';

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * Constructs a PreviewBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    StateInterface $state,
    FileUrlGeneratorInterface $file_url_generator,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->state = $state;
    $this->fileUrlGenerator = $file_url_generator;
  }

  /**
   * Build preview gallery from stored Media entities.
   *
   * @param array $settings
   *   LightGallery settings array.
   *
   * @return array
   *   Render array for the preview gallery.
   */
  public function buildPreview(array $settings) {
    $media_ids = $this->getPreviewMediaIds();

    if (empty($media_ids)) {
      return $this->buildNoMediaMessage();
    }

    return $this->buildPreviewFromMedia($media_ids, $settings);
  }

  /**
   * Build preview gallery from specific Media entity IDs.
   *
   * Uses the React LightGallery component for rendering, matching the
   * format used by the main field formatter.
   *
   * @param array $media_ids
   *   Array of Media entity IDs.
   * @param array $settings
   *   LightGallery settings array.
   *
   * @return array
   *   Render array for the preview gallery.
   */
  public function buildPreviewFromMedia(array $media_ids, array $settings) {
    $media_storage = $this->entityTypeManager->getStorage('media');
    $media_entities = $media_storage->loadMultiple($media_ids);

    if (empty($media_entities)) {
      return $this->buildNoMediaMessage();
    }

    $items = [];
    foreach ($media_entities as $media) {
      $item = $this->buildMediaItem($media);
      if ($item) {
        $items[] = $item;
      }
    }

    if (empty($items)) {
      return $this->buildNoMediaMessage();
    }

    // Generate unique gallery ID.
    $gallery_id = Html::getUniqueId('lightgallery-preview-' . uniqid());

    // Build plugin flags from settings.
    $plugin_flags = [];
    if (!empty($settings['enableThumbnail']) || !empty($settings['thumbnail'])) {
      $plugin_flags['thumbnail'] = TRUE;
    }

    // Build options from settings (filter out non-LightGallery options).
    $options = array_filter($settings, function ($key) {
      // Remove our custom flags, keep LightGallery options.
      return !in_array($key, ['enableThumbnail']);
    }, ARRAY_FILTER_USE_KEY);

    return [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['lightgallery-preview-gallery-wrapper'],
      ],
      'instructions' => [
        '#markup' => '<p class="lightgallery-preview-instructions">' .
        $this->t('Click on any image to open the gallery and test the settings.') .
        '</p>',
      ],
      'gallery' => [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#attributes' => [
          'class' => ['lightgallery-react-wrapper', 'lightgallery-preview-gallery'],
          'data-lightgallery-id' => $gallery_id,
        ],
      ],
      '#attached' => [
        'library' => [
          'lightgallery_formatter_preview/preview',
          'lightgallery_formatter/gallery',
        ],
        'drupalSettings' => [
          'lightgallery_formatter' => [
            'galleries' => [
              $gallery_id => [
                'items' => $items,
                'options' => $options,
                'plugins' => $plugin_flags,
              ],
            ],
          ],
        ],
      ],
    ];
  }

  /**
   * Build a single media item for the gallery.
   *
   * Returns item data in the React LightGallery dynamicEl format.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   *
   * @return array|null
   *   Item data array or NULL if not applicable.
   */
  protected function buildMediaItem(MediaInterface $media) {
    $source_field = $media->getSource()->getConfiguration()['source_field'] ?? NULL;

    if (!$source_field || !$media->hasField($source_field)) {
      return NULL;
    }

    $field = $media->get($source_field);
    if ($field->isEmpty()) {
      return NULL;
    }

    // Handle image media.
    if ($media->bundle() === 'image') {
      $file = $field->entity;
      if (!$file instanceof FileInterface) {
        return NULL;
      }

      $image_url = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());

      // Return in React LightGallery dynamicEl format.
      return [
        'id' => 'preview-' . $media->id(),
        'src' => $image_url,
        'thumb' => $image_url,
        'alt' => Html::escape($media->label()),
        'subHtml' => '<h4>' . Html::escape($media->label()) . '</h4>',
      ];
    }

    return NULL;
  }

  /**
   * Build message when no media is available.
   *
   * @return array
   *   Render array with message.
   */
  protected function buildNoMediaMessage() {
    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['lightgallery-preview-no-media']],
      'message' => [
        '#markup' => '<div class="messages messages--warning">' .
        $this->t('No preview media available. Please reinstall the LightGallery Formatter Preview module to create sample media.') .
        '</div>',
      ],
    ];
  }

  /**
   * Get stored preview Media entity IDs.
   *
   * @return array
   *   Array of Media entity IDs.
   */
  public function getPreviewMediaIds() {
    return $this->state->get(self::STATE_KEY, []);
  }

  /**
   * Set preview Media entity IDs in state.
   *
   * @param array $media_ids
   *   Array of Media entity IDs.
   */
  public function setPreviewMediaIds(array $media_ids) {
    $this->state->set(self::STATE_KEY, $media_ids);
  }

  /**
   * Clear preview Media entity IDs from state.
   */
  public function clearPreviewMediaIds() {
    $this->state->delete(self::STATE_KEY);
  }

}
