<?php

namespace Drupal\lightgallery_formatter\Plugin\Lightgallery;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\lightgallery_formatter\Attribute\LightgalleryPlugin;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface;
use Drupal\lightgallery_formatter\LightgalleryPluginBase;

/**
 * Provides autoplay plugin for LightGallery.
 */
#[LightgalleryPlugin(
  id: 'autoplay',
  label: new TranslatableMarkup('Autoplay'),
  description: new TranslatableMarkup('Enable automatic slideshow playback for the gallery.'),
  weight: 40,
)]
class Autoplay extends LightgalleryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultSettings(): array {
    return [
      'enabled' => FALSE,
      'autoplay' => TRUE,
      'slideShowAutoplay' => FALSE,
      'slideShowInterval' => 5000,
      'progressBar' => TRUE,
      'forceSlideShowAutoplay' => FALSE,
      'autoplayControls' => TRUE,
      'appendAutoplayControlsTo' => '.lg-toolbar',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(array $settings): bool {
    return !empty($settings['enabled']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array &$form, FormStateInterface $form_state, LightgalleryProfileInterface $profile): array {
    $settings = $this->mergeWithDefaults($profile->getPluginSettings('autoplay'));
    $plugin_id = $this->getPluginId();

    $element = [
      '#type' => 'details',
      '#title' => $this->t('Autoplay'),
      '#open' => !empty($settings['enabled']),
    ];

    $element['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable autoplay plugin'),
      '#default_value' => $settings['enabled'],
      '#description' => $this->t('Enable automatic slideshow playback for the gallery.'),
    ];

    $element['settings_wrapper'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="' . $plugin_id . '[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $element['settings_wrapper']['autoplay'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable autoplay'),
      '#default_value' => $settings['autoplay'],
      '#description' => $this->t('Enable/Disable autoplay functionality.'),
    ];

    $element['settings_wrapper']['slideShowAutoplay'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-start slideshow'),
      '#default_value' => $settings['slideShowAutoplay'],
      '#description' => $this->t('Automatically start the slideshow when gallery opens.'),
    ];

    $element['settings_wrapper']['slideShowInterval'] = [
      '#type' => 'number',
      '#title' => $this->t('Slideshow interval'),
      '#default_value' => $settings['slideShowInterval'],
      '#min' => 1000,
      '#max' => 30000,
      '#step' => 500,
      '#field_suffix' => 'ms',
      '#description' => $this->t('Time between each auto transition in milliseconds.'),
    ];

    $element['settings_wrapper']['progressBar'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show progress bar'),
      '#default_value' => $settings['progressBar'],
      '#description' => $this->t('Display a progress bar during autoplay.'),
    ];

    $element['settings_wrapper']['forceSlideShowAutoplay'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Force slideshow autoplay'),
      '#default_value' => $settings['forceSlideShowAutoplay'],
      '#description' => $this->t('Continue autoplay after user interaction. If disabled, autoplay stops after first user action.'),
    ];

    $element['settings_wrapper']['autoplayControls'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show autoplay controls'),
      '#default_value' => $settings['autoplayControls'],
      '#description' => $this->t('Display play/pause button for autoplay control.'),
    ];

    $element['settings_wrapper']['appendAutoplayControlsTo'] = [
      '#type' => 'select',
      '#title' => $this->t('Append controls to'),
      '#default_value' => $settings['appendAutoplayControlsTo'],
      '#options' => [
        '.lg-toolbar' => $this->t('Toolbar'),
        '.lg-outer' => $this->t('Gallery outer'),
      ],
      '#description' => $this->t('Where autoplay controls should be appended.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): array {
    $plugin_id = $this->getPluginId();
    $values = $form_state->getValue($plugin_id) ?? [];

    if (empty($values['enabled'])) {
      return ['enabled' => FALSE];
    }

    $settings_wrapper = $values['settings_wrapper'] ?? [];

    return [
      'enabled' => TRUE,
      'autoplay' => (bool) ($settings_wrapper['autoplay'] ?? TRUE),
      'slideShowAutoplay' => (bool) ($settings_wrapper['slideShowAutoplay'] ?? FALSE),
      'slideShowInterval' => (int) ($settings_wrapper['slideShowInterval'] ?? 5000),
      'progressBar' => (bool) ($settings_wrapper['progressBar'] ?? TRUE),
      'forceSlideShowAutoplay' => (bool) ($settings_wrapper['forceSlideShowAutoplay'] ?? FALSE),
      'autoplayControls' => (bool) ($settings_wrapper['autoplayControls'] ?? TRUE),
      'appendAutoplayControlsTo' => $settings_wrapper['appendAutoplayControlsTo'] ?? '.lg-toolbar',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildJsSettings(array $settings): array {
    if (!$this->isEnabled($settings)) {
      return [];
    }

    $merged = $this->mergeWithDefaults($settings);

    return [
      'autoplay' => (bool) $merged['autoplay'],
      'slideShowAutoplay' => (bool) $merged['slideShowAutoplay'],
      'slideShowInterval' => (int) $merged['slideShowInterval'],
      'progressBar' => (bool) $merged['progressBar'],
      'forceSlideShowAutoplay' => (bool) $merged['forceSlideShowAutoplay'],
      'autoplayControls' => (bool) $merged['autoplayControls'],
      'appendAutoplayControlsTo' => $merged['appendAutoplayControlsTo'],
    ];
  }

}
