<?php

namespace Drupal\lightgallery_formatter\Plugin\Lightgallery;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\lightgallery_formatter\Attribute\LightgalleryPlugin;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfileInterface;
use Drupal\lightgallery_formatter\LightgalleryPluginBase;

/**
 * Provides medium zoom plugin for LightGallery.
 */
#[LightgalleryPlugin(
  id: 'medium_zoom',
  label: new TranslatableMarkup('Medium Zoom'),
  description: new TranslatableMarkup('Enable Medium.com-like zoom experience for images. Note: This plugin may conflict with Toggle maximize and other resize operations. Use with caution.'),
  weight: 45,
)]
class MediumZoom extends LightgalleryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultSettings(): array {
    return [
      'enabled' => FALSE,
      'mediumZoom' => TRUE,
      'margin' => 40,
      'backgroundColor' => '#000',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(array $settings): bool {
    return !empty($settings['enabled']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array &$form, FormStateInterface $form_state, LightgalleryProfileInterface $profile): array {
    $settings = $this->mergeWithDefaults($profile->getPluginSettings('medium_zoom'));
    $plugin_id = $this->getPluginId();

    $element = [
      '#type' => 'details',
      '#title' => $this->t('Medium Zoom'),
      '#open' => !empty($settings['enabled']),
    ];

    $element['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable medium zoom plugin'),
      '#default_value' => $settings['enabled'],
      '#description' => $this->t('Enable Medium.com-like zoom experience for images. <strong>Warning:</strong> This plugin may conflict with Toggle maximize and other resize operations.'),
    ];

    $element['settings_wrapper'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="' . $plugin_id . '[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $element['settings_wrapper']['mediumZoom'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable medium zoom'),
      '#default_value' => $settings['mediumZoom'],
      '#description' => $this->t('Enable/Disable medium-like zoom experience.'),
    ];

    $element['settings_wrapper']['margin'] = [
      '#type' => 'number',
      '#title' => $this->t('Margin'),
      '#default_value' => $settings['margin'],
      '#min' => 0,
      '#max' => 200,
      '#step' => 5,
      '#field_suffix' => 'px',
      '#description' => $this->t('Space between the gallery outer area and images.'),
    ];

    $element['settings_wrapper']['backgroundColor'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Background color'),
      '#default_value' => $settings['backgroundColor'],
      '#size' => 10,
      '#maxlength' => 20,
      '#description' => $this->t('Background color for the gallery (e.g., #000, rgba(0,0,0,0.9)). Can be overwritten per item using data-lg-background-color attribute.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $plugin_id = $this->getPluginId();
    $values = $form_state->getValue($plugin_id) ?? [];

    if (!empty($values['enabled'])) {
      $settings_wrapper = $values['settings_wrapper'] ?? [];
      $margin = $settings_wrapper['margin'] ?? 40;

      if ($margin < 0 || $margin > 200) {
        $form_state->setErrorByName(
          $plugin_id . '][settings_wrapper][margin',
          $this->t('Margin must be between 0 and 200 pixels.')
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): array {
    $plugin_id = $this->getPluginId();
    $values = $form_state->getValue($plugin_id) ?? [];

    if (empty($values['enabled'])) {
      return ['enabled' => FALSE];
    }

    $settings_wrapper = $values['settings_wrapper'] ?? [];

    return [
      'enabled' => TRUE,
      'mediumZoom' => (bool) ($settings_wrapper['mediumZoom'] ?? TRUE),
      'margin' => (int) ($settings_wrapper['margin'] ?? 40),
      'backgroundColor' => $settings_wrapper['backgroundColor'] ?? '#000',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildJsSettings(array $settings): array {
    if (!$this->isEnabled($settings)) {
      return [];
    }

    $merged = $this->mergeWithDefaults($settings);

    return [
      'mediumZoom' => (bool) $merged['mediumZoom'],
      'margin' => (int) $merged['margin'],
      'backgroundColor' => $merged['backgroundColor'],
    ];
  }

}
