<?php

namespace Drupal\Tests\lightgallery_formatter\Functional;

use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests plugin form submission in LightGallery profile form.
 *
 * @group lightgallery_formatter
 */
class LightgalleryProfileFormSubmitTest extends BrowserTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * The default theme.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with admin permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->adminUser = $this->drupalCreateUser([
      'administer lightgallery_formatter',
      'administer site configuration',
    ]);
  }

  /**
   * Test creating a new profile.
   */
  public function testCreateNewProfile(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/media/lightgallery/add');

    $edit = [
      'label' => 'Test Profile',
      'id' => 'test_profile',
      'status' => TRUE,
    ];

    $this->submitForm($edit, 'Save');

    // Verify redirect and success message.
    $this->assertSession()->addressEquals('/admin/config/media/lightgallery');
    $this->assertSession()->pageTextContains('Created the Test Profile LightGallery profile.');

    // Load and verify profile.
    $profile = LightgalleryProfile::load('test_profile');
    $this->assertNotNull($profile, 'Profile created.');
    $this->assertEquals('Test Profile', $profile->label(), 'Profile label correct.');
  }

  /**
   * Test profile entity is updated on submit.
   */
  public function testProfileEntityUpdatedOnSubmit(): void {
    // Create initial profile.
    $profile = LightgalleryProfile::create([
      'id' => 'update_test',
      'label' => 'Update Test',
      'status' => TRUE,
    ]);
    $profile->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/media/lightgallery/update_test');

    $edit = [
      'label' => 'Updated Test Profile',
    ];

    $this->submitForm($edit, 'Save');

    // Verify update.
    $this->assertSession()->pageTextContains('Saved the Updated Test Profile LightGallery profile.');

    $loaded = LightgalleryProfile::load('update_test');
    $this->assertEquals('Updated Test Profile', $loaded->label(), 'Label updated.');
  }

  /**
   * Test disabled profile status saves.
   */
  public function testDisabledStatusSaves(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/media/lightgallery/add');

    $edit = [
      'label' => 'Disabled Test',
      'id' => 'disabled_test',
      'status' => FALSE,
    ];

    $this->submitForm($edit, 'Save');

    $profile = LightgalleryProfile::load('disabled_test');
    $this->assertFalse($profile->getStatus(), 'Disabled status saved.');
  }

  /**
   * Test profile list page shows profiles.
   */
  public function testProfileListPage(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/media/lightgallery');

    $this->assertSession()->statusCodeEquals(200);
    // Default profile should exist.
    $this->assertSession()->pageTextContains('Default');
  }

}
