<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;

/**
 * Tests graceful degradation when plugins are missing.
 *
 * @group lightgallery_formatter
 */
class LightgalleryFormatterGracefulDegradeTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * The plugin manager.
   *
   * @var \Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryPluginManager
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
    $this->pluginManager = $this->container->get('plugin.manager.lightgallery');
  }

  /**
   * Test profile with unknown plugin ID loads without error.
   */
  public function testProfileWithUnknownPluginLoads(): void {
    // Create profile with unknown plugin settings.
    $profile = LightgalleryProfile::create([
      'id' => 'unknown_plugin_test',
      'label' => 'Unknown Plugin Test',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => ['loop' => TRUE, 'speed' => 400],
        'nonexistent_plugin' => ['some_setting' => TRUE],
      ],
    ]);
    $profile->save();

    // Load should not throw error.
    $loaded = LightgalleryProfile::load('unknown_plugin_test');

    $this->assertNotNull($loaded, 'Profile with unknown plugin loads.');
    $this->assertEquals('Unknown Plugin Test', $loaded->label(), 'Profile label correct.');
  }

  /**
   * Test unknown plugin settings are preserved.
   */
  public function testUnknownPluginSettingsPreserved(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'preserve_test',
      'label' => 'Preserve Test',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => ['loop' => FALSE],
        'future_plugin' => ['enabled' => TRUE, 'option' => 'value'],
      ],
    ]);
    $profile->save();

    $loaded = LightgalleryProfile::load('preserve_test');
    $all_settings = $loaded->getAllPluginSettings();

    $this->assertArrayHasKey('future_plugin', $all_settings, 'Unknown plugin settings preserved.');
    $this->assertEquals(
      ['enabled' => TRUE, 'option' => 'value'],
      $all_settings['future_plugin'],
      'Unknown plugin settings values preserved.'
    );
  }

  /**
   * Test profile still provides known plugin settings correctly.
   */
  public function testKnownPluginsWorkWithUnknown(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'mixed_test',
      'label' => 'Mixed Test',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => ['loop' => FALSE, 'speed' => 800],
        'unknown_plugin' => ['foo' => 'bar'],
        'thumbnail' => ['enabled' => TRUE],
      ],
    ]);
    $profile->save();

    $loaded = LightgalleryProfile::load('mixed_test');

    // Known plugins should work normally.
    $general = $loaded->getPluginSettings('general');
    $thumbnail = $loaded->getPluginSettings('thumbnail');

    $this->assertFalse($general['loop'], 'General settings work.');
    $this->assertEquals(800, $general['speed'], 'General speed correct.');
    $this->assertTrue($thumbnail['enabled'], 'Thumbnail settings work.');
  }

  /**
   * Test iteration over plugins handles unknown gracefully.
   */
  public function testIterationHandlesUnknownGracefully(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'iterate_test',
      'label' => 'Iterate Test',
      'status' => TRUE,
      'plugin_settings' => [
        'general' => ['loop' => TRUE],
        'missing_plugin' => ['setting' => 'value'],
      ],
    ]);
    $profile->save();

    // Simulate formatter iteration: only process known plugins.
    $js_settings = [];
    $libraries = [];

    foreach ($this->pluginManager->getSortedDefinitions() as $plugin_id => $definition) {
      $plugin_settings = $profile->getPluginSettings($plugin_id);
      $plugin = $this->pluginManager->createInstance($plugin_id, $plugin_settings);

      if ($plugin->isEnabled($plugin_settings)) {
        $js_settings = array_merge($js_settings, $plugin->buildJsSettings($plugin_settings));
      }
    }

    // Should have general settings.
    $this->assertArrayHasKey('loop', $js_settings, 'General JS settings present.');
    // No errors should occur.
    $this->assertTrue(TRUE, 'Iteration completed without error.');
  }

  /**
   * Test empty plugin_settings doesn't break profile.
   */
  public function testEmptyPluginSettingsDoesntBreak(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'empty_test',
      'label' => 'Empty Test',
      'status' => TRUE,
      'plugin_settings' => [],
    ]);
    $profile->save();

    $loaded = LightgalleryProfile::load('empty_test');

    $this->assertNotNull($loaded, 'Profile with empty plugin_settings loads.');

    // Getting settings for any plugin should return empty array.
    $general = $loaded->getPluginSettings('general');
    $this->assertEmpty($general, 'Empty plugin_settings returns empty array for known plugin.');
  }

  /**
   * Test plugin defaults are used when profile has no settings.
   */
  public function testPluginDefaultsUsedWhenNoSettings(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'defaults_test',
      'label' => 'Defaults Test',
      'status' => TRUE,
      'plugin_settings' => [],
    ]);
    $profile->save();

    // Create plugin with empty settings from profile.
    $plugin = $this->pluginManager->createInstance('general', $profile->getPluginSettings('general'));

    // Plugin should use its defaults.
    $config = $plugin->getConfiguration();

    $this->assertTrue($config['loop'], 'Default loop value used.');
    $this->assertEquals(400, $config['speed'], 'Default speed value used.');
  }

  /**
   * Test malformed plugin settings don't crash.
   */
  public function testMalformedSettingsDontCrash(): void {
    // Save profile directly with non-array plugin settings (edge case).
    $profile = LightgalleryProfile::create([
      'id' => 'malformed_test',
      'label' => 'Malformed Test',
      'status' => TRUE,
    ]);
    $profile->save();

    // Force a save with edge case values.
    $loaded = LightgalleryProfile::load('malformed_test');
    $loaded->setPluginSettings('general', ['loop' => 'not_a_boolean']);
    $loaded->save();

    // Reload and ensure no crash.
    $reloaded = LightgalleryProfile::load('malformed_test');
    $this->assertNotNull($reloaded, 'Profile with malformed data loads.');

    // Plugin should handle type coercion.
    $plugin = $this->pluginManager->createInstance('general', $reloaded->getPluginSettings('general'));
    $js_settings = $plugin->buildJsSettings($reloaded->getPluginSettings('general'));

    // PHP will coerce 'not_a_boolean' to true.
    $this->assertIsBool($js_settings['loop'], 'Loop coerced to boolean.');
  }

}
