<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;

/**
 * Tests formatter runtime plugin JS settings generation.
 *
 * @group lightgallery_formatter
 */
class LightgalleryFormatterPluginRuntimeTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * The plugin manager.
   *
   * @var \Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryPluginManager
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
    $this->pluginManager = $this->container->get('plugin.manager.lightgallery');
  }

  /**
   * Test formatter builds JS settings from General plugin.
   */
  public function testFormatterBuildsJsSettingsFromGeneralPlugin(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'js_test',
      'label' => 'JS Test',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('general', [
      'loop' => FALSE,
      'speed' => 800,
      'mode' => 'lg-fade',
      'preload' => 3,
      'closable' => TRUE,
      'closeOnTap' => FALSE,
      'escKey' => TRUE,
      'keyPress' => TRUE,
      'showCloseIcon' => TRUE,
      'showMaximizeIcon' => TRUE,
      'hideBarsDelay' => 3000,
      'counter' => FALSE,
      'download' => FALSE,
      'enableDrag' => TRUE,
      'enableSwipe' => TRUE,
      'mousewheel' => TRUE,
    ]);
    $profile->save();

    $plugin = $this->pluginManager->createInstance('general', $profile->getPluginSettings('general'));
    $js_settings = $plugin->buildJsSettings($profile->getPluginSettings('general'));

    // Verify all settings are present and correct types.
    $this->assertFalse($js_settings['loop'], 'Loop JS setting is FALSE.');
    $this->assertEquals(800, $js_settings['speed'], 'Speed JS setting correct.');
    $this->assertEquals('lg-fade', $js_settings['mode'], 'Mode JS setting correct.');
    $this->assertEquals(3, $js_settings['preload'], 'Preload JS setting correct.');
    $this->assertTrue($js_settings['closable'], 'Closable JS setting correct.');
    $this->assertFalse($js_settings['closeOnTap'], 'CloseOnTap JS setting correct.');
    $this->assertTrue($js_settings['escKey'], 'EscKey JS setting correct.');
    $this->assertTrue($js_settings['keyPress'], 'KeyPress JS setting correct.');
    $this->assertTrue($js_settings['showCloseIcon'], 'ShowCloseIcon JS setting correct.');
    $this->assertTrue($js_settings['showMaximizeIcon'], 'ShowMaximizeIcon JS setting correct.');
    $this->assertEquals(3000, $js_settings['hideBarsDelay'], 'HideBarsDelay JS setting correct.');
    $this->assertFalse($js_settings['counter'], 'Counter JS setting correct.');
    $this->assertFalse($js_settings['download'], 'Download JS setting correct.');
    $this->assertTrue($js_settings['enableDrag'], 'EnableDrag JS setting correct.');
    $this->assertTrue($js_settings['enableSwipe'], 'EnableSwipe JS setting correct.');
    $this->assertTrue($js_settings['mousewheel'], 'Mousewheel JS setting correct.');
  }

  /**
   * Test formatter builds JS settings from Thumbnail plugin.
   */
  public function testFormatterBuildsJsSettingsFromThumbnailPlugin(): void {
    $settings = [
      'enabled' => TRUE,
      'animateThumb' => FALSE,
      'thumbWidth' => 120,
      'thumbHeight' => '100px',
      'thumbMargin' => 15,
      'currentPagerPosition' => 'left',
      'alignThumbnails' => 'right',
      'toggleThumb' => TRUE,
      'appendThumbnailsTo' => '.lg-outer',
    ];

    $plugin = $this->pluginManager->createInstance('thumbnail', $settings);
    $js_settings = $plugin->buildJsSettings($settings);

    $this->assertTrue($js_settings['thumbnail'], 'Thumbnail enabled in JS settings.');
    $this->assertFalse($js_settings['animateThumb'], 'AnimateThumb JS setting correct.');
    $this->assertEquals(120, $js_settings['thumbWidth'], 'ThumbWidth JS setting correct.');
    $this->assertEquals('100px', $js_settings['thumbHeight'], 'ThumbHeight JS setting correct.');
    $this->assertEquals(15, $js_settings['thumbMargin'], 'ThumbMargin JS setting correct.');
    $this->assertEquals('left', $js_settings['currentPagerPosition'], 'CurrentPagerPosition correct.');
    $this->assertEquals('right', $js_settings['alignThumbnails'], 'AlignThumbnails correct.');
    $this->assertTrue($js_settings['toggleThumb'], 'ToggleThumb JS setting correct.');
    $this->assertEquals('.lg-outer', $js_settings['appendThumbnailsTo'], 'AppendThumbnailsTo correct.');
  }

  /**
   * Test disabled thumbnail plugin returns empty JS settings.
   */
  public function testDisabledThumbnailReturnsEmptyJsSettings(): void {
    $settings = ['enabled' => FALSE];

    $plugin = $this->pluginManager->createInstance('thumbnail', $settings);
    $js_settings = $plugin->buildJsSettings($settings);

    $this->assertEmpty($js_settings, 'Disabled thumbnail returns empty JS settings.');
  }

  /**
   * Test combined settings structure is correct.
   */
  public function testCombinedSettingsStructureCorrect(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'combined_test',
      'label' => 'Combined Test',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('general', [
      'loop' => TRUE,
      'speed' => 500,
      'mode' => 'lg-slide',
      'preload' => 2,
      'closable' => TRUE,
      'closeOnTap' => TRUE,
      'escKey' => TRUE,
      'keyPress' => TRUE,
      'showCloseIcon' => TRUE,
      'showMaximizeIcon' => FALSE,
      'hideBarsDelay' => 0,
      'counter' => TRUE,
      'download' => TRUE,
      'enableDrag' => TRUE,
      'enableSwipe' => TRUE,
      'mousewheel' => FALSE,
    ]);
    $profile->setPluginSettings('thumbnail', [
      'enabled' => TRUE,
      'animateThumb' => TRUE,
      'thumbWidth' => 100,
      'thumbHeight' => '80px',
      'thumbMargin' => 10,
      'currentPagerPosition' => 'middle',
      'alignThumbnails' => 'middle',
      'toggleThumb' => FALSE,
      'appendThumbnailsTo' => '.lg-components',
    ]);
    $profile->save();

    // Collect JS settings from all plugins.
    $combined_settings = [];

    foreach ($this->pluginManager->getSortedDefinitions() as $plugin_id => $definition) {
      $plugin_settings = $profile->getPluginSettings($plugin_id);
      $plugin = $this->pluginManager->createInstance($plugin_id, $plugin_settings);

      if ($plugin->isEnabled($plugin_settings)) {
        $combined_settings = array_merge($combined_settings, $plugin->buildJsSettings($plugin_settings));
      }
    }

    // Verify combined structure.
    $this->assertArrayHasKey('loop', $combined_settings, 'Loop present in combined settings.');
    $this->assertArrayHasKey('speed', $combined_settings, 'Speed present in combined settings.');
    $this->assertArrayHasKey('thumbnail', $combined_settings, 'Thumbnail present in combined settings.');
    $this->assertArrayHasKey('animateThumb', $combined_settings, 'AnimateThumb present in combined settings.');

    // Verify values.
    $this->assertTrue($combined_settings['loop'], 'Combined loop value correct.');
    $this->assertEquals(500, $combined_settings['speed'], 'Combined speed value correct.');
    $this->assertTrue($combined_settings['thumbnail'], 'Combined thumbnail value correct.');
    $this->assertEquals(10, $combined_settings['thumbMargin'], 'Combined thumbMargin value correct.');
  }

  /**
   * Test JS settings match profile configuration.
   */
  public function testJsSettingsMatchProfileConfiguration(): void {
    $general_settings = [
      'loop' => FALSE,
      'speed' => 999,
      'mode' => 'lg-zoom-in',
      'preload' => 5,
      'closable' => FALSE,
      'closeOnTap' => FALSE,
      'escKey' => FALSE,
      'keyPress' => FALSE,
      'showCloseIcon' => FALSE,
      'showMaximizeIcon' => TRUE,
      'hideBarsDelay' => 5000,
      'counter' => FALSE,
      'download' => FALSE,
      'enableDrag' => FALSE,
      'enableSwipe' => FALSE,
      'mousewheel' => TRUE,
    ];

    $profile = LightgalleryProfile::create([
      'id' => 'match_test',
      'label' => 'Match Test',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('general', $general_settings);
    $profile->save();

    $plugin = $this->pluginManager->createInstance('general', $profile->getPluginSettings('general'));
    $js_settings = $plugin->buildJsSettings($profile->getPluginSettings('general'));

    // Verify each setting matches.
    foreach ($general_settings as $key => $expected_value) {
      $this->assertArrayHasKey($key, $js_settings, "JS settings contains '{$key}'.");
      $this->assertEquals($expected_value, $js_settings[$key], "JS setting '{$key}' matches profile.");
    }
  }

  /**
   * Test JS settings have correct types.
   */
  public function testJsSettingsHaveCorrectTypes(): void {
    $plugin = $this->pluginManager->createInstance('general');
    $js_settings = $plugin->buildJsSettings($plugin->defaultSettings());

    // Boolean settings should be boolean.
    $boolean_keys = [
      'loop',
      'closable',
      'closeOnTap',
      'escKey',
      'keyPress',
      'showCloseIcon',
      'showMaximizeIcon',
      'counter',
      'download',
      'enableDrag',
      'enableSwipe',
      'mousewheel',
    ];

    foreach ($boolean_keys as $key) {
      $this->assertIsBool($js_settings[$key], "JS setting '{$key}' is boolean.");
    }

    // Integer settings should be integer.
    $this->assertIsInt($js_settings['speed'], 'Speed is integer.');
    $this->assertIsInt($js_settings['preload'], 'Preload is integer.');
    $this->assertIsInt($js_settings['hideBarsDelay'], 'HideBarsDelay is integer.');

    // String settings should be string.
    $this->assertIsString($js_settings['mode'], 'Mode is string.');
  }

}
