<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;

/**
 * Tests multiple plugin settings storage in LightGallery profiles.
 *
 * @group lightgallery_formatter
 */
class LightgalleryProfileMultiplePluginSettingsTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['lightgallery_formatter'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
  }

  /**
   * Test saving multiple plugin configs at once.
   */
  public function testSaveMultiplePluginConfigsAtOnce(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
      'status' => TRUE,
    ]);

    // Set both plugins.
    $profile->setPluginSettings('general', [
      'loop' => FALSE,
      'speed' => 700,
      'mode' => 'lg-zoom-in',
    ]);
    $profile->setPluginSettings('thumbnail', [
      'enabled' => TRUE,
      'thumbMargin' => 15,
      'animateThumb' => FALSE,
    ]);
    $profile->save();

    // Reload and verify both are stored.
    $loaded = LightgalleryProfile::load('test_profile');

    $general = $loaded->getPluginSettings('general');
    $this->assertFalse($general['loop'], 'General loop saved correctly.');
    $this->assertEquals(700, $general['speed'], 'General speed saved correctly.');

    $thumbnail = $loaded->getPluginSettings('thumbnail');
    $this->assertTrue($thumbnail['enabled'], 'Thumbnail enabled saved correctly.');
    $this->assertEquals(15, $thumbnail['thumbMargin'], 'Thumbnail margin saved correctly.');
  }

  /**
   * Test order-independent storage.
   */
  public function testOrderIndependentStorage(): void {
    // Create with thumbnail first, then general.
    $profile1 = LightgalleryProfile::create([
      'id' => 'profile1',
      'label' => 'Profile 1',
      'status' => TRUE,
    ]);
    $profile1->setPluginSettings('thumbnail', ['enabled' => TRUE]);
    $profile1->setPluginSettings('general', ['loop' => FALSE]);
    $profile1->save();

    // Create with general first, then thumbnail.
    $profile2 = LightgalleryProfile::create([
      'id' => 'profile2',
      'label' => 'Profile 2',
      'status' => TRUE,
    ]);
    $profile2->setPluginSettings('general', ['loop' => FALSE]);
    $profile2->setPluginSettings('thumbnail', ['enabled' => TRUE]);
    $profile2->save();

    // Both should retrieve correctly regardless of order.
    $loaded1 = LightgalleryProfile::load('profile1');
    $loaded2 = LightgalleryProfile::load('profile2');

    $this->assertFalse($loaded1->getPluginSettings('general')['loop'], 'Profile1 general correct.');
    $this->assertTrue($loaded1->getPluginSettings('thumbnail')['enabled'], 'Profile1 thumbnail correct.');

    $this->assertFalse($loaded2->getPluginSettings('general')['loop'], 'Profile2 general correct.');
    $this->assertTrue($loaded2->getPluginSettings('thumbnail')['enabled'], 'Profile2 thumbnail correct.');
  }

  /**
   * Test partial updates don't overwrite other plugins.
   */
  public function testPartialUpdatePreservesOtherPlugins(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('general', [
      'loop' => TRUE,
      'speed' => 400,
    ]);
    $profile->setPluginSettings('thumbnail', [
      'enabled' => TRUE,
      'thumbMargin' => 5,
    ]);
    $profile->save();

    // Update only general settings.
    $loaded = LightgalleryProfile::load('test_profile');
    $loaded->setPluginSettings('general', [
      'loop' => FALSE,
      'speed' => 800,
    ]);
    $loaded->save();

    // Verify thumbnail settings are still intact.
    $reloaded = LightgalleryProfile::load('test_profile');
    $thumbnail = $reloaded->getPluginSettings('thumbnail');

    $this->assertTrue($thumbnail['enabled'], 'Thumbnail enabled preserved.');
    $this->assertEquals(5, $thumbnail['thumbMargin'], 'Thumbnail margin preserved.');

    // Verify general was updated.
    $general = $reloaded->getPluginSettings('general');
    $this->assertFalse($general['loop'], 'General loop updated.');
    $this->assertEquals(800, $general['speed'], 'General speed updated.');
  }

  /**
   * Test export/import preserves structure.
   */
  public function testExportImportPreservesStructure(): void {
    $original_settings = [
      'general' => [
        'loop' => FALSE,
        'speed' => 600,
        'mode' => 'lg-fade',
        'closable' => TRUE,
      ],
      'thumbnail' => [
        'enabled' => TRUE,
        'thumbMargin' => 20,
        'animateThumb' => TRUE,
      ],
    ];

    $profile = LightgalleryProfile::create([
      'id' => 'export_test',
      'label' => 'Export Test',
      'status' => TRUE,
      'plugin_settings' => $original_settings,
    ]);
    $profile->save();

    // Export to array (simulate config export).
    $exported = $profile->toArray();

    // Create new profile from exported data (simulate config import).
    // Delete original first.
    $profile->delete();

    $imported = LightgalleryProfile::create([
      'id' => $exported['id'],
      'label' => $exported['label'],
      'status' => $exported['status'],
      'plugin_settings' => $exported['plugin_settings'],
    ]);
    $imported->save();

    // Verify structure is preserved.
    $loaded = LightgalleryProfile::load('export_test');

    $this->assertEquals(
      $original_settings['general'],
      $loaded->getPluginSettings('general'),
      'General settings preserved after export/import.'
    );

    $this->assertEquals(
      $original_settings['thumbnail'],
      $loaded->getPluginSettings('thumbnail'),
      'Thumbnail settings preserved after export/import.'
    );
  }

  /**
   * Test adding new plugin settings to existing profile.
   */
  public function testAddNewPluginToExistingProfile(): void {
    // Create profile with only general settings.
    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('general', ['loop' => TRUE]);
    $profile->save();

    // Later add thumbnail settings.
    $loaded = LightgalleryProfile::load('test_profile');
    $loaded->setPluginSettings('thumbnail', ['enabled' => TRUE, 'thumbMargin' => 10]);
    $loaded->save();

    // Verify both exist.
    $reloaded = LightgalleryProfile::load('test_profile');

    $general = $reloaded->getPluginSettings('general');
    $thumbnail = $reloaded->getPluginSettings('thumbnail');

    $this->assertTrue($general['loop'], 'Original general settings preserved.');
    $this->assertTrue($thumbnail['enabled'], 'New thumbnail settings added.');
    $this->assertEquals(10, $thumbnail['thumbMargin'], 'Thumbnail margin added.');
  }

  /**
   * Test clearing plugin settings.
   */
  public function testClearingPluginSettings(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'test_profile',
      'label' => 'Test Profile',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('general', ['loop' => FALSE, 'speed' => 500]);
    $profile->setPluginSettings('thumbnail', ['enabled' => TRUE]);
    $profile->save();

    // Clear thumbnail settings by setting empty array.
    $profile->setPluginSettings('thumbnail', []);
    $profile->save();

    $loaded = LightgalleryProfile::load('test_profile');

    $general = $loaded->getPluginSettings('general');
    $thumbnail = $loaded->getPluginSettings('thumbnail');

    $this->assertNotEmpty($general, 'General settings still present.');
    $this->assertEmpty($thumbnail, 'Thumbnail settings cleared.');
  }

}
