<?php

namespace Drupal\Tests\lightgallery_formatter\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\lightgallery_formatter\Entity\LightgalleryProfile;
use Drupal\lightgallery_formatter\LightgalleryPluginInterface;

/**
 * Tests third-party plugin support for LightGallery.
 *
 * @group lightgallery_formatter
 */
class LightgalleryThirdPartyPluginTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'lightgallery_formatter',
    'lightgallery_formatter_test',
  ];

  /**
   * The plugin manager.
   *
   * @var \Drupal\lightgallery_formatter\Plugin\Lightgallery\LightgalleryPluginManager
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['lightgallery_formatter']);
    $this->pluginManager = $this->container->get('plugin.manager.lightgallery');
  }

  /**
   * Test third-party plugin discovered by manager.
   */
  public function testThirdPartyPluginDiscovered(): void {
    $definitions = $this->pluginManager->getDefinitions();

    $this->assertArrayHasKey('test_plugin', $definitions, 'Test plugin discovered by manager.');
  }

  /**
   * Test third-party plugin appears in definitions.
   */
  public function testThirdPartyPluginInDefinitions(): void {
    $definitions = $this->pluginManager->getDefinitions();
    $test_plugin = $definitions['test_plugin'];

    $this->assertEquals('test_plugin', $test_plugin['id'], 'Plugin ID is correct.');
    $this->assertNotEmpty($test_plugin['label'], 'Plugin has label.');
    $this->assertNotEmpty($test_plugin['description'], 'Plugin has description.');
    $this->assertEquals(100, $test_plugin['weight'], 'Plugin weight is correct.');
    $this->assertEquals('lightgallery_formatter_test', $test_plugin['provider'], 'Plugin provider is test module.');
  }

  /**
   * Test third-party plugin can be instantiated.
   */
  public function testThirdPartyPluginInstantiable(): void {
    $plugin = $this->pluginManager->createInstance('test_plugin');

    $this->assertNotNull($plugin, 'Test plugin instantiated.');
    $this->assertInstanceOf(LightgalleryPluginInterface::class, $plugin, 'Test plugin implements interface.');
  }

  /**
   * Test third-party plugin settings storable in profile.
   */
  public function testThirdPartyPluginSettingsStorable(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'thirdparty_test',
      'label' => 'Third Party Test',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('test_plugin', [
      'enabled' => TRUE,
      'test_option' => 'custom_value',
      'test_number' => 100,
    ]);
    $profile->save();

    $loaded = LightgalleryProfile::load('thirdparty_test');
    $settings = $loaded->getPluginSettings('test_plugin');

    $this->assertTrue($settings['enabled'], 'Test plugin enabled setting stored.');
    $this->assertEquals('custom_value', $settings['test_option'], 'Test plugin test_option stored.');
    $this->assertEquals(100, $settings['test_number'], 'Test plugin test_number stored.');
  }

  /**
   * Test third-party plugin JS settings generated.
   */
  public function testThirdPartyPluginJsSettingsGenerated(): void {
    $settings = [
      'enabled' => TRUE,
      'test_option' => 'my_value',
      'test_number' => 50,
    ];

    $plugin = $this->pluginManager->createInstance('test_plugin', $settings);
    $js_settings = $plugin->buildJsSettings($settings);

    $this->assertTrue($js_settings['testPlugin'], 'testPlugin key is TRUE.');
    $this->assertEquals('my_value', $js_settings['testOption'], 'testOption in JS settings.');
    $this->assertEquals(50, $js_settings['testNumber'], 'testNumber in JS settings.');
  }

  /**
   * Test third-party plugin disabled returns empty JS settings.
   */
  public function testThirdPartyPluginDisabledReturnsEmpty(): void {
    $settings = ['enabled' => FALSE];

    $plugin = $this->pluginManager->createInstance('test_plugin', $settings);
    $js_settings = $plugin->buildJsSettings($settings);

    $this->assertEmpty($js_settings, 'Disabled test plugin returns empty JS settings.');
  }

  /**
   * Test third-party plugin has correct default settings.
   */
  public function testThirdPartyPluginDefaultSettings(): void {
    $plugin = $this->pluginManager->createInstance('test_plugin');
    $defaults = $plugin->defaultSettings();

    $this->assertFalse($defaults['enabled'], 'Default enabled is FALSE.');
    $this->assertEquals('default_value', $defaults['test_option'], 'Default test_option correct.');
    $this->assertEquals(42, $defaults['test_number'], 'Default test_number correct.');
  }

  /**
   * Test third-party plugin sorted by weight.
   */
  public function testThirdPartyPluginSortedByWeight(): void {
    $sorted = $this->pluginManager->getSortedDefinitions();
    $plugin_ids = array_keys($sorted);

    // General (-100), Thumbnail (0), Test Plugin (100).
    $general_pos = array_search('general', $plugin_ids);
    $thumbnail_pos = array_search('thumbnail', $plugin_ids);
    $test_pos = array_search('test_plugin', $plugin_ids);

    $this->assertLessThan($thumbnail_pos, $general_pos, 'General before Thumbnail.');
    $this->assertLessThan($test_pos, $thumbnail_pos, 'Thumbnail before Test Plugin.');
    $this->assertLessThan($test_pos, $general_pos, 'General before Test Plugin.');
  }

  /**
   * Test combined settings include third-party plugin.
   */
  public function testCombinedSettingsIncludeThirdParty(): void {
    $profile = LightgalleryProfile::create([
      'id' => 'combined_thirdparty',
      'label' => 'Combined Third Party',
      'status' => TRUE,
    ]);
    $profile->setPluginSettings('general', ['loop' => FALSE]);
    $profile->setPluginSettings('test_plugin', ['enabled' => TRUE, 'test_option' => 'combined']);
    $profile->save();

    // Collect settings from all plugins.
    $combined = [];
    foreach ($this->pluginManager->getSortedDefinitions() as $plugin_id => $definition) {
      $plugin_settings = $profile->getPluginSettings($plugin_id);
      $plugin = $this->pluginManager->createInstance($plugin_id, $plugin_settings);

      if ($plugin->isEnabled($plugin_settings)) {
        $combined = array_merge($combined, $plugin->buildJsSettings($plugin_settings));
      }
    }

    // Should have both general and test plugin settings.
    $this->assertArrayHasKey('loop', $combined, 'General settings present.');
    $this->assertArrayHasKey('testPlugin', $combined, 'Test plugin settings present.');
    $this->assertEquals('combined', $combined['testOption'], 'Test option value correct.');
  }

}
