<?php

namespace Drupal\Tests\lightning_media\Functional;

use Drupal\Core\Form\FormState;
use Drupal\lightning_media\Form\SettingsForm;
use Drupal\media\Entity\Media;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests the add and edit forms for all our shipped media types.
 *
 * @group lightning_media
 */
class AddEditFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'lightning_media_audio',
    'lightning_media_document',
    'lightning_media_image',
    'lightning_media_instagram',
    'lightning_media_twitter',
    'lightning_media_video',
    'path',
  ];

  /**
   * Tests the add and edit forms for all our shipped media types.
   */
  public function testEditForms() {
    $media_types = $this->container->get('entity_type.manager')
      ->getStorage('media_type')
      ->getQuery()
      ->execute();

    $permissions = [
      'create url aliases',
    ];
    foreach ($media_types as $media_type) {
      $permissions[] = "create $media_type media";
      $permissions[] = "edit own $media_type media";
    }
    $account = $this->drupalCreateUser($permissions);
    $this->drupalLogin($account);

    $assert_session = $this->assertSession();
    // List of media type which are required to have oembed data or embed code.
    // Using the dummy data to test the add and edit pages.
    $media_oembed = [
      'remote_video' => 'https://www.youtube.com/watch?v=Nf_aD3dTloY',
      'tweet' => 'https://twitter.com/50NerdsofGrey/status/757319527151636480',
      'instagram' => 'https://www.instagram.com/p/BmIh_AFDBzX',
    ];

    $existing_media = [];
    foreach ($media_types as $media_type) {
      $this->drupalGet("/media/add/$media_type");
      $assert_session->statusCodeEquals(200);
      $assert_session->fieldNotExists('URL alias');
      $assert_session->fieldNotExists('Revision log message');

      $media = Media::create([
        'bundle' => $media_type,
        'name' => $this->randomString(),
      ]);

      if (isset($media_oembed[$media_type])) {
        $source_field = $media->getSource()
          ->getSourceFieldDefinition($media->bundle->entity)
          ->getName();

        $media->set($source_field, $media_oembed[$media_type]);
      }
      $media->setPublished()->save();
      $this->drupalGet($media->toUrl('edit-form'));
      $assert_session->statusCodeEquals(200);
      $assert_session->fieldNotExists('Create new revision');
      $assert_session->fieldNotExists('Revision log message');
      array_push($existing_media, $media);
    }

    // Enable the revision UI via our settings form. To speed things up, just
    // directly execute the form without logging in through the UI.
    $form_state = new FormState();
    $form_state->setValue('revision_ui', TRUE);
    $this->container->get('form_builder')
      ->submitForm(SettingsForm::class, $form_state);

    // The revision UI should now be visible, but the revision log message is
    // still hidden in the form display.
    foreach ($existing_media as $media) {
      $this->drupalGet($media->toUrl('edit-form'));
      $assert_session->statusCodeEquals(200);
      $assert_session->fieldExists('Create new revision');
      $assert_session->fieldNotExists('Revision log message');
    }

    foreach ($media_types as $media_type) {
      $this->container->get('entity_display.repository')
        ->getFormDisplay('media', $media_type)
        ->setComponent('revision_log_message')
        ->save();
    }

    foreach ($existing_media as $media) {
      $this->drupalGet($media->toUrl('edit-form'));
      $assert_session->statusCodeEquals(200);
      $assert_session->fieldExists('Create new revision');
      $assert_session->fieldExists('Revision log message');
    }
  }

}
