<?php

namespace Drupal\Tests\lingotek\Unit\Plugin\LingotekFormComponent\Filter;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\PagerSelectExtender;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\lingotek\LingotekConfigurationServiceInterface;
use Drupal\lingotek\LingotekContentTranslationServiceInterface;
use Drupal\lingotek\Plugin\LingotekFormComponent\Filter\TargetStatus;
use Drupal\Tests\UnitTestCase;

/**
 * Unit test for the target status query filter form component.
 *
 * @coversDefaultClass \Drupal\lingotek\Plugin\LingotekFormComponent\Filter\TargetStatus
 * @group lingotek
 * @preserve GlobalState disabled
 */
class TargetStatusTest extends UnitTestCase {

  /**
   * The class instance under test.
   *
   * @var \Drupal\lingotek\Plugin\LingotekFormComponent\Filter\SourceLanguage
   */
  protected $filter;

  /**
   * The connection object on which to run queries.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $connection;

  /**
   * The mocked language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $languageManager;

  /**
   * The mocked entity type.
   *
   * @var \Drupal\Core\Entity\ContentEntityTypeInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityType;

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The mocked entity_type.bundle.info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeBundleInfo;

  /**
   * The mocked Lingotek configuration service.
   *
   * @var \Drupal\lingotek\LingotekConfigurationServiceInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $lingotekConfiguration;

  /**
   * The mocked Lingotek content translation service.
   *
   * @var \Drupal\lingotek\LingotekContentTranslationServiceInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $lingotekContentTranslation;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->connection = $this->createMock(Connection::class);

    $this->entityType = $this->createMock(ContentEntityTypeInterface::class);
    $this->entityType->expects($this->any())
      ->method('getBundleLabel')
      ->willReturn(new TranslatableMarkup("My Bundle Label"));
    $this->entityType->expects($this->any())
      ->method('getKey')
      ->willReturnCallback(function ($argument) {
        switch ($argument) {
          case 'id':
            return 'id';

          case 'langcode':
            return 'langcode';

          case 'bundle':
            return 'bundle';

          default:
            return NULL;
        }
      });

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->entityTypeBundleInfo = $this->createMock(EntityTypeBundleInfoInterface::class);
    $this->languageManager = $this->createMock(LanguageManagerInterface::class);
    $this->lingotekConfiguration = $this->createMock(LingotekConfigurationServiceInterface::class);
    $this->lingotekContentTranslation = $this->createMock(LingotekContentTranslationServiceInterface::class);
    $this->filter = new TargetStatus([], 'target_status', ['id' => 'target_status', 'title' => 'Target Status'], $this->entityTypeManager, $this->entityTypeBundleInfo, $this->languageManager, $this->lingotekConfiguration, $this->lingotekContentTranslation, $this->connection);

    /** @var \Drupal\Core\StringTranslation\TranslationInterface $translation */
    $translation = $this->getStringTranslationStub();
    $this->filter->setStringTranslation($translation);
  }

  /**
   * @covers ::isApplicable
   */
  public function testIsApplicable() {
    $arguments = [
      'entity_type_id' => 'an_entity_type',
    ];
    $this->assertTrue($this->filter->isApplicable($arguments));
  }

  /**
   * @covers ::getSubmittedValue
   */
  public function testGetSubmittedValue() {
    $value = ['wrapper' => ['target_status' => 'CURRENT']];
    $this->assertEquals('CURRENT', $this->filter->getSubmittedValue($value));
  }

  /**
   * @covers ::buildElement
   */
  public function testBuildElement() {
    $this->filter->setEntityTypeId('my_entity_type_id');
    $build = $this->filter->buildElement();
    $this->assertEquals([
      '#type' => 'select',
      '#title' => 'Target Status',
      '#default_value' => '',
      '#options' => [
        '' => 'All',
        'CURRENT' => 'Current',
        'EDITED' => 'Edited',
        'PENDING' => 'In Progress',
        'READY' => 'Ready',
        'INTERMEDIATE' => 'Interim',
        'REQUEST' => 'Not Requested',
        'CANCELLED' => 'Cancelled',
        'ERROR' => 'Error',
        'LOCKED' => 'Locked',
      ],
    ],
      $build);
  }

  /**
   * @covers ::filter
   */
  public function testFilter() {
    $entity_type = $this->createMock(ContentEntityTypeInterface::class);
    $entity_type->expects($this->any())
      ->method('id')
      ->willReturn('my_entity_type');
    $entity_type->expects($this->any())
      ->method('getKey')
      ->willReturnCallback(function ($argument) {
        switch ($argument) {
          case 'id':
            return 'entity_id';

          case 'langcode':
            return 'langcode';

          default:
            return NULL;
        }
      });
    $entity_type->expects($this->any())
      ->method('getBaseTable')
      ->willReturn('entity_datatable');
    $metadata = $this->createMock(ContentEntityTypeInterface::class);
    $metadata->expects($this->any())
      ->method('getBaseTable')
      ->willReturn('metadata_content');
    $this->entityTypeManager->expects($this->exactly(2))
      ->method('getDefinition')
      ->willReturnCallback(function ($entityTypeId) use ($entity_type, $metadata) {
        if ($entityTypeId === 'my_entity_type_id') {
            return $entity_type;
        }
        elseif ($entityTypeId === 'lingotek_content_metadata') {
          return $metadata;
        }
        return NULL;
      });

    $subquery = $this->createMock(PagerSelectExtender::class);
    $this->connection->expects($this->once())
      ->method('select')
      ->with('entity_datatable', 'entity_table')
      ->willReturn($subquery);
    $subquery->expects(($this->once()))
      ->method('extend')
      ->with('\Drupal\Core\Database\Query\PagerSelectExtender')
      ->willReturnSelf();
    $subquery->expects(($this->once()))
      ->method('fields')
      ->with('entity_table', ['entity_id'])
      ->willReturnSelf();

    $subquery->method('innerJoin')
      ->willReturnCallback(function () use ($subquery) {
        return $subquery;
      });
    $subquery->innerJoin(
      $this->equalTo('metadata_content'),
      $this->equalTo('metadata_target'),
      $this->equalTo("entity_table.entity_id= metadata_target.content_entity_id AND metadata_target.content_entity_type_id = 'my_entity_type_id'")
      );

    $subquery->innerJoin(
      $this->equalTo('lingotek_content_metadata__translation_status'),
      $this->equalTo('translation_target_status'),
      $this->equalTo('metadata_target.id = translation_target_status.entity_id AND translation_target_status.translation_status_language <> entity_table.langcode')
      );
    $subquery->expects($this->once())
      ->method('condition')
      ->with('translation_target_status.translation_status_value', ['CURRENT'], '=')
      ->willReturnSelf();

    $unionQuery = $this->createMock(PagerSelectExtender::class);

    $select = $this->createMock(PagerSelectExtender::class);
    $select->expects($this->once())
      ->method('condition')
      ->with('entity_table.entity_id', $subquery, 'IN')
      ->willReturnSelf();
    $select->expects($this->once())
      ->method('getUnion')
      ->willReturn([
        ['query' => $unionQuery],
      ]);
    $unionQuery->expects($this->once())
      ->method('condition')
      ->with('entity_table.entity_id', $subquery, 'IN')
      ->willReturnSelf();

    $this->filter->filter('my_entity_type_id', [], ['CURRENT'], $select);
  }

}
