<?php

namespace Drupal\Tests\lingotek\Functional;

use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\language\Entity\ContentLanguageSettings;
use Drupal\Tests\content_moderation\Traits\ContentModerationTestTrait;
use Drupal\Tests\taxonomy\Traits\TaxonomyTestTrait;

/**
 * Tests setting up the integration with content moderation.
 *
 * @group lingotek
 */
class LingotekContentModerationSettingsTest extends LingotekTestBase {

  use TaxonomyTestTrait;
  use ContentModerationTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['node', 'taxonomy', 'content_moderation'];

  /**
   * Vocabulary for testing.
   *
   * @var \Drupal\taxonomy\VocabularyInterface
   */
  protected $vocabulary;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->drupalCreateContentType(['type' => 'article', 'name' => 'Article']);
    $this->drupalCreateContentType(['type' => 'page', 'name' => 'Page']);

    $this->vocabulary = $this->createVocabulary();

    // Add a language.
    ConfigurableLanguage::createFromLangcode('es')
      ->setThirdPartySetting('lingotek', 'locale', 'es_MX')
      ->save();

    // Enable translation for the current entity type and ensure the change is
    // picked up.
    ContentLanguageSettings::loadByEntityTypeBundle('node', 'article')->setLanguageAlterable(TRUE)->save();
    \Drupal::service('content_translation.manager')->setEnabled('node', 'article', TRUE);

    ContentLanguageSettings::loadByEntityTypeBundle('node', 'page')->setLanguageAlterable(TRUE)->save();
    \Drupal::service('content_translation.manager')->setEnabled('node', 'page', TRUE);

    ContentLanguageSettings::loadByEntityTypeBundle('taxonomy_term', $this->vocabulary->id())->setLanguageAlterable(TRUE)->save();
    \Drupal::service('content_translation.manager')->setEnabled('taxonomy_term', $this->vocabulary->id(), TRUE);

    drupal_static_reset();
    \Drupal::entityTypeManager()->clearCachedDefinitions();
    $this->applyEntityUpdates();
    // Rebuild the container so that the new languages are picked up by services
    // that hold a list of languages.
    $this->rebuildContainer();

    $workflow = $this->createEditorialWorkflow();
  }

  /**
   * Tests that the content moderation state entity cannot be enabled for Lingotek.
   */
  public function testContentModerationStateEntityNotAvailableForLingotek() {
    $assert_session = $this->assertSession();

    ContentLanguageSettings::loadByEntityTypeBundle('content_moderation_state', 'content_moderation_state')->setLanguageAlterable(TRUE)->save();
    \Drupal::service('content_translation.manager')->setEnabled('content_moderation_state', 'content_moderation_state', TRUE);

    $this->drupalGet('admin/lingotek/settings');

    $this->assertSession()->fieldNotExists('content_moderation_state[content_moderation_state][enabled]');

    // See content_moderation_entity_bundle_info_alter().
    $this->enableModerationThroughUI('article', ['draft', 'needs_review', 'published'], 'draft');
    $this->saveLingotekContentTranslationSettingsForNodeTypes();

    $this->drupalGet('admin/lingotek/settings');

    $this->assertSession()->fieldNotExists('content_moderation_state[content_moderation_state][enabled]');
  }

  /**
   * Tests that the content moderation settings are stored correctly.
   */
  public function testContentModerationSettings() {
    $assert_session = $this->assertSession();

    $this->drupalGet('admin/lingotek/settings');

    // We don't have any fields for configuring content moderation until it's
    // enabled.
    $this->assertSession()->fieldNotExists('node[article][moderation][upload_status]');
    $this->assertSession()->fieldNotExists('node[article][moderation][download_transition]');

    $this->assertSession()->fieldNotExists('node[page][moderation][upload_status]');
    $this->assertSession()->fieldNotExists('node[page][moderation][download_transition]');

    // We show a message and link for enabling it.
    $this->assertSession()->pageTextContains('This entity bundle is not enabled for moderation with content_moderation. You can change its settings here.');

    $assert_session->linkByHrefExists('/admin/config/workflow/workflows', 0);
    $assert_session->linkByHrefExists('/admin/config/workflow/workflows', 1);

    // Let's enable it for articles.
    $this->enableModerationThroughUI('article', ['draft', 'needs_review', 'published'], 'draft');

    $this->drupalGet('admin/lingotek/settings');

    // Assert the fields for setting up the integration exist and they have
    // sensible defaults.
    $this->assertSession()->fieldExists('node[article][moderation][upload_status]');
    $this->assertSession()->fieldExists('node[article][moderation][download_transition]');
    $assert_session->optionExists('edit-node-article-moderation-upload-status', 'published');
    $assert_session->optionExists('edit-node-article-moderation-download-transition', 'publish');

    // But not for the other content types. There is still a message for configuring.
    $this->assertSession()->fieldNotExists('node[page][moderation][upload_status]');
    $this->assertSession()->fieldNotExists('node[page][moderation][download_transition]');
    $this->assertSession()->pageTextContains('This entity bundle is not enabled for moderation with content_moderation. You can change its settings here.');

    $assert_session->linkByHrefExists('/admin/config/workflow/workflows', 0);

    // Let's save the settings for articles.
    $this->saveLingotekContentTranslationSettings([
      'node' => [
        'article' => [
          'profiles' => 'automatic',
          'fields' => [
            'title' => 1,
          ],
          'moderation' => [
            'upload_status' => 'draft',
            'download_transition' => 'archive',
          ],
        ],
      ],
    ]);

    // Assert the values are saved.
    $assert_session->optionExists('edit-node-article-moderation-upload-status', 'draft');
    $assert_session->optionExists('edit-node-article-moderation-download-transition', 'archive');

    // It never existed for taxonomies.
    $this->assertSession()->fieldNotExists("taxonomy_term[{$this->vocabulary->id()}][moderation][upload_status]");
    $this->assertSession()->fieldNotExists("taxonomy_term[{$this->vocabulary->id()}][moderation][download_transition]");
    $assert_session->linkByHrefNotExists("/admin/structure/taxonomy/manage/{$this->vocabulary->id()}/moderation", 'There is no link to moderation settings in taxonomies as they cannot be moderated.');

    $header = $this->xpath("//details[@id='edit-entity-node']//th[text()='Content moderation']");
    $this->assertEquals(count($header), 1, 'There is a content moderation column for content.');
    $header = $this->xpath("//details[@id='edit-entity-taxonomy-term']//th[text()='Content moderation']");
    $this->assertEquals(count($header), 0, 'There is no content moderation column for taxonomies.');

  }

  /**
   * Enable moderation for a specified content type, using the UI.
   *
   * @param string $content_type_id
   *   Machine name.
   */
  protected function enableModerationThroughUI($content_type_id) {
    $this->drupalGet('/admin/config/workflow/workflows/manage/editorial/type/node');
    $this->assertSession()->fieldExists("bundles[$content_type_id]");
    $edit["bundles[$content_type_id]"] = TRUE;
    $this->submitForm($edit, t('Save'));
  }

}
