<?php

declare(strict_types=1);

namespace Drupal\link_orcid\Form;

use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for Link an ORCID.
 */
final class LinkOrcidSettingsForm extends ConfigFormBase implements ContainerInjectionInterface {

  /**
   * {@inheritdoc}
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typed_config_manager,
    private readonly EntityFieldManagerInterface $entityFieldManager,
    private readonly KeyRepositoryInterface $keyRepository,
  ) {
    parent::__construct($configFactory, $typed_config_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_field.manager'),
      $container->get('key.repository'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'link_orcid_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['link_orcid.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('link_orcid.settings');

    $form['permission_info'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--warning"><strong>Important:</strong> To allow users to link their ORCID, grant the <em>link own orcid</em> permission to your users in <a href="/admin/people/permissions/module/link_orcid">People &gt; Permissions</a>.</div>',
      '#weight' => -100,
    ];

    // Gather user text fields to store the ORCID iD.
    $definitions = $this->entityFieldManager->getFieldDefinitions('user', 'user');
    $options = [];
    foreach ($definitions as $field_name => $definition) {
      // Allow any string-based field types: string, string_long.
      if (in_array($definition->getType(), ['string', 'string_long'], TRUE)) {
        $label = $definition->getLabel();
        $options[$field_name] = $label ? ($label . " ($field_name)") : $field_name;
      }
    }

    $form['user_field'] = [
      '#type' => 'select',
      '#title' => $this->t('User field for ORCID'),
      '#description' => $this->t('Choose a plain text field on the User entity to store the ORCID iD.'),
      '#options' => $options,
      '#required' => TRUE,
      '#default_value' => $config->get('user_field') ?: NULL,
    ];

    $form['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('ORCID Client ID'),
      '#required' => TRUE,
      '#default_value' => $config->get('client_id') ?: '',
    ];

    // Key module: pick a key to hold the client secret.
    $key_options = [];
    foreach ($this->keyRepository->getKeys() as $key) {
      $key_options[$key->id()] = $key->label();
    }
    $form['secret_key'] = [
      '#type' => 'select',
      '#title' => $this->t('Key containing ORCID Client Secret'),
      '#description' => $this->t('Select the Key that stores your ORCID client secret. You must create this key first using the <a href=":link">Key module</a>.', [':link' => '/admin/config/system/keys']),
      '#options' => $key_options,
      '#required' => TRUE,
      '#default_value' => $config->get('secret_key') ?: NULL,
    ];

    $form['sandbox'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use ORCID Sandbox'),
      '#description' => $this->t('Enable to use the ORCID sandbox environment for testing.'),
      '#default_value' => (bool) $config->get('sandbox'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);
    $this->config('link_orcid.settings')
      ->set('user_field', $form_state->getValue('user_field'))
      ->set('client_id', trim((string) $form_state->getValue('client_id')))
      ->set('secret_key', $form_state->getValue('secret_key'))
      ->set('sandbox', (bool) $form_state->getValue('sandbox'))
      ->save();
  }

}
