<?php

namespace Drupal\Tests\link_orcid\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field\Entity\FieldConfig;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Core\Entity\Entity\EntityViewDisplay;

/**
 * Functional test for Link ORCID module.
 *
 * @group link_orcid
 */
class LinkOrcidTest extends BrowserTestBase {

  /**
   * The machine name of the ORCID field.
   *
   * @var string
   */
  protected $orcidFieldName = 'field_orcid';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create the ORCID field on the user entity if it does not exist.
    $field_name = $this->orcidFieldName;
    $entity_type = 'user';
    $bundle = 'user';

    // Create field storage if it doesn't exist.
    if (!\Drupal::entityTypeManager()->getStorage('field_storage_config')->load($entity_type . '.' . $field_name)) {
      FieldStorageConfig::create([
        'field_name' => $field_name,
        'entity_type' => $entity_type,
        'type' => 'string',
        'cardinality' => 1,
        'settings' => [
          'max_length' => 19,
        ],
      ])->save();
    }

    // Create field instance if it doesn't exist.
    if (!\Drupal::entityTypeManager()->getStorage('field_config')->load($entity_type . '.' . $bundle . '.' . $field_name)) {
      FieldConfig::create([
        'field_name' => $field_name,
        'entity_type' => $entity_type,
        'bundle' => $bundle,
        'label' => 'ORCID',
        'required' => FALSE,
      ])->save();
    }

    // Add field to user form display.
    $form_display = EntityFormDisplay::load($entity_type . '.' . $bundle . '.default');
    if ($form_display) {
      $form_display->setComponent($field_name, [
        'type' => 'string_textfield',
        'weight' => 10,
      ])->save();
    }

    // Add field to user view display.
    $view_display = EntityViewDisplay::load($entity_type . '.' . $bundle . '.default');
    if ($view_display) {
      $view_display->setComponent($field_name, [
        'type' => 'string',
        'weight' => 10,
      ])->save();
    }

    // Set the field in the module config.
    \Drupal::service('config.factory')->getEditable('link_orcid.settings')->set('user_field', $field_name)->save();
  }

  /**
   * Test unlinking ORCID from user profile.
   */
  public function testUnlinkOrcid() {
    $user = $this->drupalCreateUser(['link own orcid']);
    $this->drupalLogin($user);

    $field_name = $this->orcidFieldName;
    $user->set($field_name, '0001-0002-0003-0004');
    $user->save();

    // Go to profile edit page and confirm Unlink button is present.
    $this->drupalGet('user/' . $user->id() . '/edit');
    $this->assertSession()->pageTextContains('Unlink ORCID');

    // Click the Unlink button (simulate request).
    $this->drupalGet('/link-orcid/unlink');
    $this->assertSession()->pageTextContains('Your ORCID has been unlinked.');

    // Reload user and confirm field is empty.
    $user = \Drupal::entityTypeManager()->getStorage('user')->load($user->id());
    $this->assertTrue(empty($user->{$field_name}->value));
  }

  /**
   * Use the Stark theme for functional tests.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['user', 'field', 'key', 'link_orcid'];

  /**
   * Test ORCID link button appears for authenticated users.
   */
  public function testOrcidButtonAppears() {
    // Create a user with permission.
    $user = $this->drupalCreateUser(['link own orcid']);
    $this->drupalLogin($user);

    // Go to profile edit page.
    $this->drupalGet('user/' . $user->id() . '/edit');
    $this->assertSession()->pageTextContains('Link ORCID');
  }

  /**
   * Test config form displays permission info message.
   */
  public function testConfigFormPermissionMessage() {
    $admin = $this->drupalCreateUser(['administer link orcid']);
    $this->drupalLogin($admin);
    $this->drupalGet('admin/config/people/link-orcid');
    $this->assertSession()->pageTextContains('grant the link own orcid permission');
  }

}
