<?php

namespace Drupal\link_param_propagator\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Link Param Propagator tracking rules.
 */
class TrackingConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['link_param_propagator_tracking.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'link_param_propagator_tracking_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('link_param_propagator_tracking.settings');

    // Use posted values on rebuild; otherwise use saved config.
    $entries = $form_state->getValue('rules') ?? $config->get('rules') ?? [];

    if ($entries === NULL) {
      $entries = $form_state->get('rules_draft');
    }
    if ($entries === NULL) {
      $entries = $config->get('rules');
    }
    if (!$entries) {
      $entries = [['label' => '', 'tracking_params' => '', 'target_selector' => '']];
    }
    $form_state->set('rules_draft', $entries);

    $form['rules'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Tracking Rules'),
      '#tree' => TRUE,
      '#prefix' => '<div id="link-param-rules-wrapper">',
      '#suffix' => '</div>',
    ];

    $rules_count = $form_state->get('rules_count');
    if ($rules_count === NULL) {
      $rules_count = max(1, count($entries));
      $form_state->set('rules_count', $rules_count);
    }

    for ($i = 0; $i < $rules_count; $i++) {
      $label = !empty($entries[$i]['label'])
        ? $entries[$i]['label']
        : $this->t('Rule @num', ['@num' => $i + 1]);

      $form['rules'][$i] = [
        '#type' => 'fieldset',
        '#title' => $label,
      ];

      $form['rules'][$i]['label'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Rule Label'),
        '#description' => $this->t('Enter a descriptive name for this rule (e.g., Homepage UTM Tracking).'),
        '#default_value' => $entries[$i]['label'] ?? '',
      ];

      $form['rules'][$i]['tracking_params'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Tracking Parameters'),
        '#description' => $this->t('Comma-separated parameters, e.g., utm_source, utm_medium'),
        '#default_value' => isset($entries[$i]['tracking_params'])
          ? (is_array($entries[$i]['tracking_params'])
            ? implode(', ', $entries[$i]['tracking_params'])
            : $entries[$i]['tracking_params'])
          : '',
      ];

      $form['rules'][$i]['target_selector'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Target DOM Selector'),
        '#description' => $this->t('e.g., #main-content, .lp-testimonials'),
        '#default_value' => $entries[$i]['target_selector'] ?? '',
      ];

      // Delete button for this rule.
      $form['rules'][$i]['actions']['remove'] = [
        '#type' => 'submit',
        '#value' => $this->t('Delete'),
        '#name' => "remove_rule_$i",
        '#submit' => ['::removeRuleSubmit'],
        '#limit_validation_errors' => [],
        '#ajax' => [
          'callback' => '::ajaxRulesWrapper',
          'wrapper' => 'link-param-rules-wrapper',
        ],
        '#attributes' => ['class' => ['button', 'button--danger']],
      ];
    }

    // Add another rule button.
    $form['add_rule'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add another rule'),
      '#submit' => ['::addRuleSubmit'],
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => '::ajaxRulesWrapper',
        'wrapper' => 'link-param-rules-wrapper',
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * AJAX callback for the rules wrapper.
   */
  public function ajaxRulesWrapper(array &$form, FormStateInterface $form_state) {
    return $form['rules'];
  }

  /**
   * Add another rule submit handler.
   */
  public function addRuleSubmit(array &$form, FormStateInterface $form_state) {
    $count = (int) $form_state->get('rules_count');
    $form_state->set('rules_count', $count + 1);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Remove a rule submit handler.
   */
  public function removeRuleSubmit(array &$form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement();
    if (!empty($trigger['#name']) && preg_match('/remove_rule_(\d+)/', $trigger['#name'], $m)) {
      $idx = (int) $m[1];

      // Read raw POST, not processed values (processed only contains the clicked button).
      $input = $form_state->getUserInput();
      $entries = isset($input['rules']) && is_array($input['rules']) ? $input['rules'] : [];

      // Fallback to any draft we stored earlier.
      if (!$entries) {
        $entries = $form_state->get('rules_draft') ?? [];
      }

      unset($entries[$idx]);
      $entries = array_values($entries);

      // Ensure at least one empty row remains.
      if (!$entries) {
        $entries = [['label' => '', 'tracking_params' => '', 'target_selector' => '']];
      }

      // Persist for rebuild.
      $form_state->setValue('rules', $entries);
      $form_state->set('rules_draft', $entries);
      $form_state->set('rules_count', count($entries));
      $form_state->setRebuild(TRUE);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValue('rules') ?? [];
    $rules = [];

    foreach ($values as $entry) {
      $params = array_filter(array_map('trim',
        is_array($entry['tracking_params'])
          ? $entry['tracking_params']
          : explode(',', (string) $entry['tracking_params'])
      ), 'strlen');

      $rules[] = [
        'label' => trim((string) ($entry['label'] ?? '')),
        'tracking_params' => $params,
        'target_selector' => trim((string) ($entry['target_selector'] ?? '')),
      ];
    }

    $this->config('link_param_propagator_tracking.settings')
      ->set('rules', $rules)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
