<?php

namespace Drupal\links_action_ui\Form;

use Drupal\Component\Uuid\Uuid;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Form handler for the Local Action add and edit forms.
 */
class LocalActionForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);

    /** @var \Drupal\links_action_ui\Entity\LocalAction $local_action */
    $local_action = $this->entity;

    // Initialize appears_on_routes in form state if not exists.
    if (!$form_state->has('appears_on_routes')) {
      $form_state->set('appears_on_routes', $local_action->getAppearsOnUrls() ?: ['']);
    }
    $routes = $form_state->get('appears_on_routes');

    $form['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Button Text'),
      '#maxlength' => 255,
      '#default_value' => $local_action->getTitle(),
      '#description' => $this->t("The text displayed on the action button."),
      '#required' => TRUE,
    ];

    $form['url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Url'),
      '#default_value' => $local_action->getUrl(),
      '#description' => $this->t('Enter url: /admin/xxx'),
      '#required' => TRUE,
    ];

    // Container for route list.
    $form['appears_on_container'] = [
      '#type' => 'fieldset',
      '#prefix' => '<div id="appears-on-container">',
      '#suffix' => '</div>',
      '#title' => $this->t('Append on'),
      '#tree' => TRUE,
    ];

    foreach ($routes as $key => $route) {
      $form['appears_on_container'][$key] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['container-inline', 'mb-2']],
      ];

      $form['appears_on_container'][$key]['route'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Route'),
        '#title_display' => 'invisible',
        '#default_value' => $route,
        '#required' => TRUE,
        '#placeholder' => $this->t('Enter url: /admin/xxx'),
        '#size' => 60,
      ];

      // Delete button for route.
      if (count($routes) > 1) {
        $form['appears_on_container'][$key]['remove'] = [
          '#type' => 'submit',
          '#value' => $this->t('Remove'),
          '#name' => 'remove_route_' . $key,
          '#submit' => ['::removeRoute'],
          '#ajax' => [
            'callback' => '::ajaxRefreshRoutes',
            'wrapper' => 'appears-on-container',
          ],
          '#limit_validation_errors' => [],
          '#attributes' => ['class' => ['btn-danger', 'ml-2']],
        ];
      }
    }

    // Button to add another route.
    $form['add_route'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add another route'),
      '#submit' => ['::addRoute'],
      '#ajax' => [
        'callback' => '::ajaxRefreshRoutes',
        'wrapper' => 'appears-on-container',
      ],
      '#limit_validation_errors' => [],
      '#attributes' => ['class' => ['btn-secondary', 'mt-2']],
    ];

    $form['weight'] = [
      '#type' => 'weight',
      '#title' => $this->t('Weight'),
      '#default_value' => $local_action->getWeight(),
      '#description' => $this->t('Actions with lower weights appear first.'),
    ];

    return $form;
  }

  /**
   * AJAX callback: Refresh route list.
   */
  public function ajaxRefreshRoutes(array &$form, FormStateInterface $form_state) {
    return $form['appears_on_container'];
  }

  /**
   * Add a new route.
   */
  public function addRoute(array &$form, FormStateInterface $form_state) {
    $routes = $form_state->get('appears_on_routes');
    $routes[] = '';
    $form_state->set('appears_on_routes', $routes);
    $form_state->setRebuild();
  }

  /**
   * Remove a route.
   */
  public function removeRoute(array &$form, FormStateInterface $form_state) {
    $routes = $form_state->get('appears_on_routes');
    $key = str_replace('remove_route_', '', $form_state->getTriggeringElement()['#name']);
    
    if (isset($routes[$key])) {
      unset($routes[$key]);
      $form_state->set('appears_on_routes', array_values($routes));
    }
    $form_state->setRebuild();
  }

  /**
   * Validate url format and route existence.
   * 
   * @param string $url
   *   The URL string to validate.
   * 
   * @throws \Exception
   *   Throws exception if URL is invalid.
   * 
   * @return \Drupal\Core\Url
   *   The validated Url object.
   */
  public function validateUrl(string $url) {
    if (!str_starts_with($url, '/')) {
        throw new \Exception($this->t('Url must start with /: @url', [
            '@url' => $url,
        ]));
    }
    $urlObj = Url::fromUserInput($url);

    if (!$urlObj->isRouted()) {
        throw new \Exception($this->t('The URL is not a valid route: @url', [
            '@url' => $url,
        ]));
    }

    return $urlObj;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate target URL.
    $url = $form_state->getValue('url');
    $append_on = $form_state->getValue('appears_on_container');

    try {
        $urlObj = $this->validateUrl($url);
        $form_state->setValue('route_name', $urlObj->getRouteName());
        $form_state->setValue('route_parameters', $urlObj->getRouteParameters());
    } catch(\Exception $e) {
        return $form_state->setErrorByName('url', $e->getMessage());
    }

    $append_on_routes = [];
    foreach($append_on as $item) {
        try {
            $itemObj = $this->validateUrl($item['route']);
            $append_on_routes[] = $itemObj->getRouteName();
        } catch(\Exception $e) {
            return $form_state->setErrorByName('appears_on_container', $e->getMessage());
        }
    }
    $form_state->setValue('appears_on_urls', array_column($append_on, 'route'));
    $form_state->setValue('appears_on', $append_on_routes);
  }

  /**
   * save entity.
   * 
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   * @return void
   */
  public function save(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\links_action_ui\Entity\LocalAction $local_action */
    $local_action = $this->entity;

    // Generate UUID for new entities.
    if ($local_action->isNew()) {
        $local_action->set('id', \Drupal::service('uuid')->generate());
    }
    $status = $local_action->save();

    $message_params = ['%label' => $local_action->label()];
    if ($status === SAVED_NEW) {
      $this->messenger()->addStatus($this->t('Created new local action: %label.', $message_params));
    }
    else {
      $this->messenger()->addStatus($this->t('Updated local action: %label.', $message_params));
    }

    // Clear local action plugin cache to apply changes.
    \Drupal::service('plugin.manager.menu.local_action')->clearCachedDefinitions();

    $form_state->setRedirectUrl($local_action->toUrl('collection'));
  }

}
