<?php

namespace Drupal\links_action_ui\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Form handler for the Local Action add and edit forms.
 */
class LocalActionForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);

    /** @var \Drupal\links_action_ui\Entity\LocalAction $local_action */
    $local_action = $this->entity;

    // Initialize appears_on_routes in form state if not exists.
    if (!$form_state->has('appears_on_routes')) {
      $form_state->set('appears_on_routes', $local_action->getAppearsOn() ?: ['']);
    }
    $routes = $form_state->get('appears_on_routes');

    // Initialize route parameters.
    if (!$form_state->has('route_parameters')) {
       // Ensure we have at least one empty parameter set
      $params = [['key' => '', 'value' => '']];
      if ($sys_params = $local_action->getRouteParameters() ?: []) {
        $params = [];
        foreach($sys_params as $key => $value) {
          $params[] = [
            'key' => $key,
            'value' => $value
          ];
        }
      }
      $form_state->set('route_parameters', $params);
    }
    $parameters = $form_state->get('route_parameters');

    $form['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Button Text'),
      '#maxlength' => 255,
      '#default_value' => $local_action->getTitle(),
      '#description' => $this->t("The text displayed on the action button."),
      '#required' => TRUE,
    ];

    $form['route_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Route name'),
      '#default_value' => $local_action->getRouteName(),
      '#description' => $this->t('Enter route name'),
      '#required' => TRUE,
    ];

    $form['route_parameters_parent'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Route Parameters'),
    ];
    // Route parameters fieldset
    $form['route_parameters_parent']['route_parameters_container'] = [
      '#type' => 'container',
      '#prefix' => '<div id="route-parameters-container">',
      '#suffix' => '</div>',
      '#description' => $this->t('Add custom parameters to be included in the route (e.g., destination=123)'),
      '#tree' => TRUE,
    ];

    foreach ($parameters as $key => $param) {
      $form['route_parameters_parent']['route_parameters_container'][$key] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['container-inline', 'mb-2']],
      ];

      $form['route_parameters_parent']['route_parameters_container'][$key]['key'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Parameter Key'),
        '#title_display' => 'invisible',
        '#default_value' => $param['key'],
        '#placeholder' => $this->t('e.g., destination'),
        '#size' => 20,
      ];

      $form['route_parameters_parent']['route_parameters_container'][$key]['value'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Parameter Value'),
        '#title_display' => 'invisible',
        '#default_value' => $param['value'],
        '#placeholder' => $this->t('e.g., 123'),
        '#size' => 20,
        '#attributes' => ['class' => ['ml-2']],
      ];

      // Delete button for parameter
      if (count($parameters) > 1) {
        $form['route_parameters_parent']['route_parameters_container'][$key]['remove'] = [
          '#type' => 'submit',
          '#value' => $this->t('Remove'),
          '#name' => 'remove_param_' . $key,
          '#submit' => ['::removeParameter'],
          '#ajax' => [
            'callback' => '::ajaxRefreshParameters',
            'wrapper' => 'route-parameters-container',
          ],
          '#limit_validation_errors' => [],
          '#attributes' => ['class' => ['btn-danger', 'ml-2']],
        ];
      }
    }

    // Button to add another parameter
    $form['route_parameters_parent']['add_parameter'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add parameter'),
      '#submit' => ['::addParameter'],
      '#ajax' => [
        'callback' => '::ajaxRefreshParameters',
        'wrapper' => 'route-parameters-container',
      ],
      '#limit_validation_errors' => [],
      '#attributes' => ['class' => ['btn-secondary', 'mt-2']],
    ];

    $form['appears_on_parent'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Append on'),
    ];
    // Container for route list.
    $form['appears_on_parent']['appears_on_container'] = [
      '#type' => 'container',
      '#prefix' => '<div id="appears-on-container">',
      '#suffix' => '</div>',
      '#tree' => TRUE,
    ];

    foreach ($routes as $key => $route) {
      $form['appears_on_parent']['appears_on_container'][$key] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['container-inline', 'mb-2']],
      ];

      $form['appears_on_parent']['appears_on_container'][$key]['route'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Route'),
        '#title_display' => 'invisible',
        '#default_value' => $route,
        '#required' => TRUE,
        '#placeholder' => $this->t('Enter route name'),
        '#size' => 60,
      ];

      // Delete button for route.
      if (count($routes) > 1) {
        $form['appears_on_parent']['appears_on_container'][$key]['remove'] = [
          '#type' => 'submit',
          '#value' => $this->t('Remove'),
          '#name' => 'remove_route_' . $key,
          '#submit' => ['::removeRoute'],
          '#ajax' => [
            'callback' => '::ajaxRefreshRoutes',
            'wrapper' => 'appears-on-container',
          ],
          '#limit_validation_errors' => [],
          '#attributes' => ['class' => ['btn-danger', 'ml-2']],
        ];
      }
    }

    // Button to add another route.
    $form['appears_on_parent']['add_route'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add append on'),
      '#submit' => ['::addRoute'],
      '#ajax' => [
        'callback' => '::ajaxRefreshRoutes',
        'wrapper' => 'appears-on-container',
      ],
      '#limit_validation_errors' => [],
      '#attributes' => ['class' => ['btn-secondary', 'mt-2']],
    ];

    $form['weight'] = [
      '#type' => 'weight',
      '#title' => $this->t('Weight'),
      '#default_value' => $local_action->getWeight(),
      '#description' => $this->t('Actions with lower weights appear first.'),
    ];

    return $form;
  }

  /**
   * AJAX callback: Refresh parameters list.
   */
  public function ajaxRefreshParameters(array &$form, FormStateInterface $form_state) {
    return $form['route_parameters_parent']['route_parameters_container'];
  }

  /**
   * Add a new parameter.
   */
  public function addParameter(array &$form, FormStateInterface $form_state) {
    $params = $form_state->get('route_parameters');
    $params[] = ['key' => '', 'value' => ''];
    $form_state->set('route_parameters', $params);
    $form_state->setRebuild();
  }

  /**
   * Remove a parameter.
   */
  public function removeParameter(array &$form, FormStateInterface $form_state) {
    $params = $form_state->get('route_parameters');
    $key = str_replace('remove_param_', '', $form_state->getTriggeringElement()['#name']);

    if (isset($params[$key])) {
      unset($params[$key]);
      $form_state->set('route_parameters', array_values($params));
    }
    $form_state->setRebuild();
  }

  /**
   * AJAX callback: Refresh route list.
   */
  public function ajaxRefreshRoutes(array &$form, FormStateInterface $form_state) {
    return $form['appears_on_parent']['appears_on_container'];
  }

  /**
   * Add a new route.
   */
  public function addRoute(array &$form, FormStateInterface $form_state) {
    $routes = $form_state->get('appears_on_routes');
    $routes[] = '';
    $form_state->set('appears_on_routes', $routes);
    $form_state->setRebuild();
  }

  /**
   * Remove a route.
   */
  public function removeRoute(array &$form, FormStateInterface $form_state) {
    $routes = $form_state->get('appears_on_routes');
    $key = str_replace('remove_route_', '', $form_state->getTriggeringElement()['#name']);

    if (isset($routes[$key])) {
      unset($routes[$key]);
      $form_state->set('appears_on_routes', array_values($routes));
    }
    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    $service = \Drupal::service('router.route_provider');

    // build route parameters.
    $route_parameters = [];
    foreach($form_state->getValue('route_parameters_container') as $item) {
      $route_parameters[$item['key']] = $item['value'];
    }
    $append_on = [];
    if ($form_state->getValue('appears_on_container')) {
      $append_on = array_column($form_state->getValue('appears_on_container'), 'route');
    }
    // validate route name.
    try {
      $service->getRouteByName($form_state->getValue('route_name'));
    } catch(\Exception $e) {
      return $form_state->setErrorByName('route_name', t('The route name is unavailable: @route', [
        '@route' => $form_state->getValue('route_name')
      ]));
    }


    // validate appears_on route name.
    foreach($append_on as $item) {
      try {
        $service->getRouteByName($item);
      } catch(\Exception $e) {
        return $form_state->setErrorByName('appears_on_container', t('The route name is unavailable: @route', [
          '@route' => $item
        ]));
      }
    }

    $form_state->setValue('route_name', $form_state->getValue('route_name'));
    $form_state->setValue('route_parameters', $route_parameters);
    $form_state->setValue('appears_on', $append_on);
  }

  /**
   * save entity.
   * 
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   * @return void
   */
  public function save(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\links_action_ui\Entity\LocalAction $local_action */
    $local_action = $this->entity;

    // Generate UUID for new entities.
    if ($local_action->isNew()) {
      $local_action->set('id', \Drupal::service('uuid')->generate());
    }
    $status = $local_action->save();

    $message_params = ['%label' => $local_action->label()];
    if ($status === SAVED_NEW) {
      $this->messenger()->addStatus($this->t('Created new local action: %label.', $message_params));
    }
    else {
      $this->messenger()->addStatus($this->t('Updated local action: %label.', $message_params));
    }

    // Clear local action plugin cache to apply changes.
    \Drupal::service('plugin.manager.menu.local_action')->clearCachedDefinitions();

    $form_state->setRedirectUrl($local_action->toUrl('collection'));
  }

}
